<?php
require_once __DIR__ . '/config.php';
require_once __DIR__ . '/functions.php';
require_once __DIR__ . '/time.php';
require_once __DIR__ . '/commands.php';
require_once __DIR__ . '/member.php';
require_once __DIR__ . '/pv_menu.php';

function handleUpdate($update) {
    if (!empty($update['message'])) {
        handleMessage($update['message']);
    } elseif (!empty($update['callback_query'])) {
        handleCallback($update['callback_query']);
    }
}

function handleMessage($msg) {
    $chat = $msg['chat'] ?? [];
    $from = $msg['from'] ?? [];
    $text = $msg['text'] ?? '';
    $chat_id = $chat['id'] ?? null;
    $from_id = $from['id'] ?? null;
    $msg_id = $msg['message_id'] ?? null;
    if (!$chat_id || !$from_id || !$msg_id) return;

    if (hasProcessedUpdate('msg', $msg_id)) return;

    cacheUser($from_id, $from['first_name'] ?? 'کاربر', $from['last_name'] ?? null, $from['username'] ?? null);

    if ($chat['type'] === 'private') {
        if ($text === '/start' || $text === '/start@' . BOT_USERNAME) {
            showMainMenu($from_id, $chat_id);
            return;
        }
        if (preg_match('/^([A-Z0-9\-]{8,20})$/i', trim($text), $m)) {
            $code = strtoupper(trim($m[1]));
            $stmt = db()->prepare("SELECT code FROM activation_codes WHERE code = ? AND used_by_chat_id IS NULL");
            $stmt->execute([$code]);
            if ($stmt->fetch()) {
                sendTelegram('sendMessage', [
                    'chat_id' => $chat_id,
                    'text' => "✅ <b>کد صحیح است!</b>\n<code>$code</code>\n\n1. ربات را به گروه اضافه کنید.\n2. ربات را ادمین کنید.\n3. دستور <code>فعال‌سازی $code</code> را در گروه بفرستید.",
                    'parse_mode' => 'HTML'
                ]);
            } else {
                sendTelegram('sendMessage', ['chat_id' => $chat_id, 'text' => "❌ کد نامعتبر یا قبلاً استفاده شده است."]);
            }
            return;
        }
        return;
    }

    if (in_array($chat['type'], ['group', 'supergroup'])) {
        recordMessage($chat_id, $from_id, $msg_id);
        
        if (preg_match('/^([A-Z0-9\-]{8,20})$/i', trim($text), $m)) {
            cmd_activate_with_code($chat_id, $from_id, strtoupper(trim($m[1])), $msg_id);
            return;
        }
        if (preg_match('/^فعال\s*سازی\s+([A-Z0-9\-]+)$/iu', $text, $m)) {
            cmd_activate_with_code($chat_id, $from_id, strtoupper(trim($m[1])), $msg_id);
            return;
        }
        if (preg_match('/^\/activate\s+([A-Z0-9\-]+)$/i', $text, $m)) {
            cmd_activate_with_code($chat_id, $from_id, strtoupper(trim($m[1])), $msg_id);
            return;
        }
        if ((int)$from_id === OWNER_ID && in_array($text, ['فعال‌سازی', '/activate'])) {
            cmd_activate_manual($chat_id, $msg_id);
            return;
        }
        if (!isGroupActive($chat_id)) return;
        
        if (preg_match('/^(قفل|بازکردن)\s+(.+)$/iu', $text, $m)) {
            handleTextLockCommand($chat_id, $from_id, $m[1], $m[2], $msg_id);
            return;
        }

        if (handleMemberCommands($chat_id, $from_id, $text, $msg_id, $msg['reply_to_message'] ?? null)) return;
        if (handleCommands($chat_id, $from_id, $text, $msg_id, $msg['reply_to_message'] ?? null)) return;
        handleContent($msg);
    }
}

function handleTextLockCommand($chat_id, $user_id, $action, $type, $msg_id) {
    if (!canManageLocks($chat_id, $user_id)) {
        sendMessageWithAutoDelete($chat_id, "❌ شما دسترسی تنظیم قفل‌ها را ندارید.", $msg_id);
        return;
    }

    $lock_types = [
        'لینک' => 'link', 'استیکر' => 'sticker', 'عکس' => 'photo', 'ویدیو' => 'video',
        'فایل' => 'document', 'صدا' => 'voice', 'گیف' => 'animation', 'مخاطب' => 'contact',
        'لوکیشن' => 'location', 'فوروارد' => 'forward'
    ];
    
    $type_cleaned = trim($type);
    $type_en = $lock_types[$type_cleaned] ?? null;

    if (!$type_en) {
        $available = implode('، ', array_keys($lock_types));
        sendMessageWithAutoDelete($chat_id, "❌ نوع قفل نامعتبر است.\n\nانواع موجود: $available", $msg_id);
        return;
    }

    $new_state = ($action === 'قفل');
    setLock($chat_id, $type_en, $new_state);
    
    $status = $new_state ? "🔒 قفل شد" : "🔓 باز شد";
    sendMessageWithAutoDelete($chat_id, "✅ {$type_cleaned} $status", $msg_id);
}

function handleContent($msg) {
    $chat_id = $msg['chat']['id'];
    $from_id = $msg['from']['id'];
    $msg_id = $msg['message_id'];
    $role = getUserRole($chat_id, $from_id);
    
    // 🔥 مالکان، ادمین‌ها و کاربران ویژه از قفل‌ها معافند
    if (in_array($role, ['owner_bot', 'owner_group', 'owner_assigned', 'vip']) || (is_array($role) && $role['role'] === 'admin')) {
        return;
    }

    sendExpiryWarningIfNeeded($chat_id);
    
    // بررسی قفل لینک
    if (isLocked($chat_id, 'link')) {
        $has_link = !empty($msg['text']) && preg_match('/(https?:\/\/|t\.me\/|telegram\.me\/|www\.)/i', $msg['text']);
        $has_link = $has_link || (!empty($msg['caption']) && preg_match('/(https?:\/\/|t\.me\/|telegram\.me\/|www\.)/i', $msg['caption']));
        if (!$has_link && !empty($msg['entities'])) {
            foreach ($msg['entities'] as $e) {
                if (in_array($e['type'], ['url', 'text_link', 'mention'])) { $has_link = true; break; }
            }
        }
        if (!$has_link && !empty($msg['caption_entities'])) {
            foreach ($msg['caption_entities'] as $e) {
                if (in_array($e['type'], ['url', 'text_link'])) { $has_link = true; break; }
            }
        }
        if ($has_link) {
            sendTelegram('deleteMessage', ['chat_id' => $chat_id, 'message_id' => $msg_id]);
            return;
        }
    }
    
    // بررسی سایر قفل‌ها
    $types = ['sticker','photo','video','document','voice','animation','contact','location','forward'];
    foreach ($types as $type) {
        $has = match($type) {
            'sticker' => !empty($msg['sticker']),
            'photo' => !empty($msg['photo']),
            'video' => !empty($msg['video']),
            'document' => !empty($msg['document']),
            'voice' => !empty($msg['voice']),
            'animation' => !empty($msg['animation']),
            'contact' => !empty($msg['contact']),
            'location' => !empty($msg['location']),
            'forward' => (!empty($msg['forward_from']) || !empty($msg['forward_from_chat']) || !empty($msg['forward_date'])),
        };
        if ($has && isLocked($chat_id, $type)) {
            sendTelegram('deleteMessage', ['chat_id' => $chat_id, 'message_id' => $msg_id]);
            return;
        }
    }
}

function handleCallback($query) {
    $data = $query['data'] ?? '';
    $from = $query['from'] ?? [];
    $user_id = $from['id'] ?? null;
    $message = $query['message'] ?? [];
    $chat = $message['chat'] ?? [];
    $chat_id = $chat['id'] ?? null;
    $msg_id = $message['message_id'] ?? null;
    $callback_id = $query['id'];
    
    if (!$user_id || !$chat_id || !$data) return;
    
    if (hasProcessedUpdate('cbq', $callback_id)) {
        sendTelegram('answerCallbackQuery', ['callback_query_id' => $callback_id]);
        return;
    }

    if (preg_match('/^(tag|adm|lock)_(.+)$/', $data, $match)) {
        $prefix = $match[1];
        $action_hash = $match[2];
        
        if ($prefix === 'lock') {
            handleLockCallback($query);
            return;
        }
        
        if (preg_match('/^(close|cancel)_(.{16})$/', $action_hash, $close_match)) {
            $hash = $close_match[2];
            $panel_json = getCache("panel_$hash");
            
            if (!$panel_json) {
                sendTelegram('answerCallbackQuery', ['callback_query_id' => $callback_id, 'text' => "❌ پنل منقضی شده.", 'show_alert' => true]);
                sendTelegram('deleteMessage', ['chat_id' => $chat_id, 'message_id' => $msg_id]);
                return;
            }
            
            $panel_data = json_decode($panel_json, true);
            if ((string)$user_id !== (string)($panel_data['issuer_id'] ?? '')) {
                sendTelegram('answerCallbackQuery', ['callback_query_id' => $callback_id, 'text' => "🔒 این پنل فقط برای فرستنده است.", 'show_alert' => true]);
                return;
            }
            
            db()->prepare("DELETE FROM cache WHERE k = ?")->execute(["panel_$hash"]);
            sendTelegram('deleteMessage', ['chat_id' => $chat_id, 'message_id' => $msg_id]);
            sendTelegram('answerCallbackQuery', ['callback_query_id' => $callback_id, 'text' => "✅ پنل بسته شد."]);
            return;
        }
        
        if (!preg_match('/^([a-z_]+)_(.{16})$/', $action_hash, $parts)) {
            sendTelegram('answerCallbackQuery', ['callback_query_id' => $callback_id]);
            return;
        }
        
        $action = $parts[1];
        $hash = $parts[2];

        $panel_json = getCache("panel_$hash");
        if (!$panel_json) {
            sendTelegram('answerCallbackQuery', ['callback_query_id' => $callback_id, 'text' => "❌ پنل منقضی شده.", 'show_alert' => true]);
            sendTelegram('deleteMessage', ['chat_id' => $chat_id, 'message_id' => $msg_id]);
            return;
        }

        $panel_data = json_decode($panel_json, true);
        if ((string)$user_id !== (string)($panel_data['issuer_id'] ?? '')) {
            sendTelegram('answerCallbackQuery', ['callback_query_id' => $callback_id, 'text' => "🔒 این پنل فقط برای فرستنده است.", 'show_alert' => true]);
            return;
        }
        
        if ($prefix === 'tag') {
            handleTagCallbackAction($query, $panel_data, $action, $hash);
            return;
        }

        if ($prefix === 'adm') {
            handleAdminCallbackAction($query, $panel_data, $action, $hash);
            return;
        }
    }

    if (preg_match('/^panel_refresh_(\d+)$/', $data, $pm) && (int)$user_id === OWNER_ID) {
        $total = db()->query("SELECT COUNT(*) FROM groups")->fetchColumn();
        $active = db()->query("SELECT COUNT(*) FROM groups WHERE expires_at > NOW()")->fetchColumn();
        sendTelegram('editMessageText', [
            'chat_id' => $chat_id, 'message_id' => $msg_id,
            'text' => "💎 <b>پنل مدیریت کل</b>\n\n📊 آمار:\n• کل گروه‌ها: $total\n• فعال: $active",
            'parse_mode' => 'HTML',
            'reply_markup' => ['inline_keyboard' => [[['text' => '🔄 رفرش', 'callback_data' => "panel_refresh_{$user_id}"]]]]
        ]);
        sendTelegram('answerCallbackQuery', ['callback_query_id' => $callback_id, 'text' => "🔄 اطلاعات بروزرسانی شد"]);
        return;
    }

    if ($chat['type'] === 'private' && strpos($data, 'pv_') === 0) {
        handlePvCallback($query); 
        return; 
    }
}

function handleTagCallbackAction($query, $panel_data, $action, $hash) {
    $chat_id = $query['message']['chat']['id'];
    $callback_id = $query['id'];
    $msg_id = $query['message']['message_id'];
    
    sendTelegram('answerCallbackQuery', ['callback_query_id' => $callback_id, 'text' => "✅ در حال اجرا..."]);
    
    if ($action === 'members') {
        tagAllMembers($chat_id, $panel_data['reply_to_msg_id']);
    } elseif ($action === 'admins') {
        tagAllAdmins($chat_id, $panel_data['reply_to_msg_id']);
    }
    
    db()->prepare("DELETE FROM cache WHERE k = ?")->execute(["panel_$hash"]);
    sendTelegram('deleteMessage', ['chat_id' => $chat_id, 'message_id' => $msg_id]);
}

function handleAdminCallbackAction($query, $panel_data, $action, $hash) {
    $chat_id = $query['message']['chat']['id'];
    $msg_id = $query['message']['message_id'];
    $callback_id = $query['id'];
    $target_id = $panel_data['target_id'];
    $stored_chat_id = $panel_data['chat_id'];

    if ($action === 'cancel') {
        sendTelegram('deleteMessage', ['chat_id' => $chat_id, 'message_id' => $msg_id]);
        sendTelegram('answerCallbackQuery', ['callback_query_id' => $callback_id, 'text' => "❌ لغو شد"]);
        db()->prepare("DELETE FROM cache WHERE k = ?")->execute(["panel_$hash"]);
        return;
    }

    if (in_array($action, ['ban', 'mute', 'delete', 'pin', 'vip'])) {
        $perm_key = "can_$action";
        $panel_data[$perm_key] = !($panel_data[$perm_key] ?? false);
        setCache("panel_$hash", json_encode($panel_data), 600);

        $ban_icon = ($panel_data['can_ban'] ?? false) ? "✅" : "❌";
        $mute_icon = ($panel_data['can_mute'] ?? false) ? "✅" : "❌";
        $delete_icon = ($panel_data['can_delete'] ?? false) ? "✅" : "❌";
        $pin_icon = ($panel_data['can_pin'] ?? false) ? "✅" : "❌";
        $settings_icon = ($panel_data['can_vip'] ?? false) ? "✅" : "❌";
        
        $keyboard = ['inline_keyboard' => [
            [['text' => "🚫 بن $ban_icon", 'callback_data' => "adm_ban_$hash"]],
            [['text' => "🔇 سکوت $mute_icon", 'callback_data' => "adm_mute_$hash"]],
            [['text' => "🗑 حذف پیام $delete_icon", 'callback_data' => "adm_delete_$hash"]],
            [['text' => "📌 سنجاق $pin_icon", 'callback_data' => "adm_pin_$hash"]],
            [['text' => "⚙️ تنظیمات $settings_icon", 'callback_data' => "adm_vip_$hash"]],
            [['text' => '✅ تایید و ذخیره', 'callback_data' => "adm_save_$hash"]],
            [['text' => '❌ انصراف', 'callback_data' => "adm_cancel_$hash"]]
        ]];
        
        $target_mention = formatUserMention($target_id);
        sendTelegram('editMessageText', [
            'chat_id' => $chat_id, 'message_id' => $msg_id,
            'text' => "🛠 <b>تنظیم دسترسی‌های مدیر</b>\n\n👤 کاربر: $target_mention\n\nدسترسی‌ها را با کلیک بر روی دکمه‌ها تنظیم کنید:",
            'parse_mode' => 'HTML', 'reply_markup' => $keyboard
        ]);
        
        $perm_names = ['ban' => 'بن', 'mute' => 'سکوت', 'delete' => 'حذف پیام', 'pin' => 'سنجاق', 'vip' => 'تنظیمات'];
        $status_text = ($panel_data[$perm_key] ?? false) ? "فعال ✅" : "غیرفعال ❌";
        sendTelegram('answerCallbackQuery', ['callback_query_id' => $callback_id, 'text' => "{$perm_names[$action]}: $status_text"]);
        return;
    }

    if ($action === 'save') {
        try {
            $can_ban = ($panel_data['can_ban'] ?? false) ? 1 : 0;
            $can_mute = ($panel_data['can_mute'] ?? false) ? 1 : 0;
            $can_delete = ($panel_data['can_delete'] ?? false) ? 1 : 0;
            $can_pin = ($panel_data['can_pin'] ?? false) ? 1 : 0;
            $can_vip = ($panel_data['can_vip'] ?? false) ? 1 : 0;
            
            db()->prepare("INSERT INTO bot_admins (chat_id, user_id, can_ban, can_mute, can_delete, can_pin, can_vip, added_by) VALUES (?, ?, ?, ?, ?, ?, ?, ?) ON DUPLICATE KEY UPDATE can_ban = ?, can_mute = ?, can_delete = ?, can_pin = ?, can_vip = ?")->execute([
                $stored_chat_id, $target_id, $can_ban, $can_mute, $can_delete, $can_pin, $can_vip, $panel_data['issuer_id'],
                $can_ban, $can_mute, $can_delete, $can_pin, $can_vip
            ]);
            
            $target_mention = formatUserMention($target_id);
            $perms_text = "✅ <b>مدیر با موفقیت ثبت شد</b>\n\n👤 کاربر: $target_mention\n\n📋 <b>دسترسی‌ها:</b>\n";
            $perms_text .= "• بن: " . ($can_ban ? "✅" : "❌") . "\n";
            $perms_text .= "• سکوت: " . ($can_mute ? "✅" : "❌") . "\n";
            $perms_text .= "• حذف پیام: " . ($can_delete ? "✅" : "❌") . "\n";
            $perms_text .= "• سنجاق: " . ($can_pin ? "✅" : "❌") . "\n";
            $perms_text .= "• تنظیمات: " . ($can_vip ? "✅" : "❌");
            
            sendTelegram('editMessageText', [
                'chat_id' => $chat_id, 'message_id' => $msg_id,
                'text' => $perms_text, 'parse_mode' => 'HTML', 'reply_markup' => ['inline_keyboard' => []]
            ]);
            
            sendTelegram('answerCallbackQuery', ['callback_query_id' => $callback_id, 'text' => "✅ تنظیمات با موفقیت ذخیره شد"]);
            scheduleMessageDeletion($chat_id, $msg_id, 120);
            db()->prepare("DELETE FROM cache WHERE k = ?")->execute(["panel_$hash"]);
        } catch (Exception $e) {
            log_debug("خطا در ذخیره مدیر: " . $e->getMessage());
            sendTelegram('answerCallbackQuery', ['callback_query_id' => $callback_id, 'text' => "❌ خطا در ذخیره تنظیمات", 'show_alert' => true]);
        }
    }
}

function handleLockCallback($query) {
    $data = $query['data'];
    $user_id = $query['from']['id'];
    $chat_id = $query['message']['chat']['id'];
    $msg_id = $query['message']['message_id'];
    $callback_id = $query['id'];

    if (preg_match('/^lock_([a-z]+)_?(.{16})?$/', $data, $m)) {
        $type = $m[1];
        $hash = $m[2] ?? null;
        
        if($hash) {
            $panel_json = getCache("panel_$hash");
            if(!$panel_json) {
                sendTelegram('answerCallbackQuery', ['callback_query_id' => $callback_id, 'text' => "❌ پنل منقضی شده.", 'show_alert' => true]);
                sendTelegram('deleteMessage', ['chat_id' => $chat_id, 'message_id' => $msg_id]);
                return;
            }
            $panel_info = json_decode($panel_json, true);
            if((string)$user_id !== (string)($panel_info['issuer_id'] ?? '')) {
                 sendTelegram('answerCallbackQuery', ['callback_query_id' => $callback_id, 'text' => "🔒 این پنل فقط برای فرستنده است.", 'show_alert' => true]);
                 return;
            }
        } else {
            if (!canManageLocks($chat_id, $user_id)) {
                 sendTelegram('answerCallbackQuery', ['callback_query_id' => $callback_id, 'text' => "❌ دسترسی ندارید.", 'show_alert' => true]);
                 return;
            }
        }
        
        if ($type === 'done') { 
            sendTelegram('deleteMessage', ['chat_id' => $chat_id, 'message_id' => $msg_id]); 
            sendTelegram('answerCallbackQuery', ['callback_query_id' => $callback_id, 'text' => "✅ پنل بسته شد."]);
            if($hash) {
                db()->prepare("DELETE FROM cache WHERE k = ?")->execute(["panel_$hash"]);
            }
            return; 
        }
        
        $new_state = !isLocked($chat_id, $type);
        setLock($chat_id, $type, $new_state);
        
        showLockMenu($chat_id, $msg_id, true, $msg_id, $hash);
        
        $type_names = [
            'link' => 'لینک', 'sticker' => 'استیکر', 'photo' => 'عکس', 'video' => 'ویدیو',
            'document' => 'فایل', 'voice' => 'صدا', 'animation' => 'GIF', 'contact' => 'مخاطب',
            'location' => 'لوکیشن', 'forward' => 'فوروارد'
        ];
        
  $type_fa = $type_names[$type] ?? $type;
        $status = $new_state ? "🔒 قفل شد" : "🔓 باز شد";
        sendTelegram('answerCallbackQuery', ['callback_query_id' => $callback_id, 'text' => "$type_fa $status"]);
    }
}

function recordMessage($chat_id, $user_id, $msg_id) {
    try {
        db()->prepare("INSERT INTO messages (chat_id, user_id, message_id, sent_at) VALUES (?, ?, ?, NOW())")->execute([$chat_id, $user_id, $msg_id]);
    } catch (PDOException $e) {
        if ($e->getCode() !== 23000) {
            log_debug("Error recording message: " . $e->getMessage());
        }
    }
}

function cmd_activate_with_code($chat_id, $user_id, $code, $msg_id) {
    $stmt = db()->prepare("SELECT id, duration_days FROM activation_codes WHERE code = ? AND used_by_chat_id IS NULL");
    $stmt->execute([$code]);
    $row = $stmt->fetch();
    if (!$row) {
        sendMessageWithAutoDelete($chat_id, "❌ کد نامعتبر یا قبلاً استفاده شده است.", $msg_id);
        return;
    }

    $owner_id = getGroupOwnerFromChat($chat_id);
    if (!$owner_id) {
        sendMessageWithAutoDelete($chat_id, "⚠️ ربات باید ادمین کامل باشد.", $msg_id);
        return;
    }

    $days = (int)$row['duration_days'];
    $expire_at = date('Y-m-d H:i:s', time() + ($days * 86400));

    db()->prepare("INSERT INTO groups (chat_id, owner_id, expires_at, activated_at) VALUES (?, ?, ?, NOW()) ON DUPLICATE KEY UPDATE expires_at = ?, last_expiry_warning = NULL")
        ->execute([$chat_id, $owner_id, $expire_at, $expire_at]);

    db()->prepare("UPDATE activation_codes SET used_by_chat_id = ?, used_at = NOW() WHERE id = ?")
        ->execute([$chat_id, $row['id']]);

    db()->prepare("INSERT IGNORE INTO bot_admins (chat_id, user_id, can_ban, can_mute, can_delete, can_pin, can_vip, added_by) VALUES (?, ?, 1, 1, 1, 1, 1, ?)")
        ->execute([$chat_id, $owner_id, OWNER_ID]);

    $jalali = jdate('Y/m/d', strtotime($expire_at));
    sendTelegram('sendMessage', [
        'chat_id' => $chat_id,
        'text' => "🎉 گروه فعال شد!\n\n📅 مدت: $days روز\n📆 انقضا: $jalali\n👑 مالک: " . formatUserMention($owner_id),
        'parse_mode' => 'HTML',
        'reply_to_message_id' => $msg_id
    ]);

    sendTelegram('sendMessage', [
        'chat_id' => OWNER_ID,
        'text' => "💰 کد $code استفاده شد.\n📱 گروه: $chat_id\n👤 توسط: " . formatUserMention($user_id),
        'parse_mode' => 'HTML'
    ]);
}

function cmd_activate_manual($chat_id, $msg_id) {
    $owner_id = getGroupOwnerFromChat($chat_id);
    if (!$owner_id) {
        sendMessageWithAutoDelete($chat_id, "❌ ربات ادمین نیست.", $msg_id);
        return;
    }

    $expire_at = date('Y-m-d H:i:s', time() + (30 * 86400));

    db()->prepare("INSERT INTO groups (chat_id, owner_id, expires_at, activated_at) VALUES (?, ?, ?, NOW()) ON DUPLICATE KEY UPDATE expires_at = ?, last_expiry_warning = NULL")
        ->execute([$chat_id, $owner_id, $expire_at, $expire_at]);

    db()->prepare("INSERT IGNORE INTO bot_admins (chat_id, user_id, can_ban, can_mute, can_delete, can_pin, can_vip, added_by) VALUES (?, ?, 1, 1, 1, 1, 1, ?)")
        ->execute([$chat_id, $owner_id, OWNER_ID]);

    $jalali = jdate('Y/m/d', strtotime($expire_at));
    sendTelegram('sendMessage', [
        'chat_id' => $chat_id,
        'text' => "✅ فعال‌سازی دستی انجام شد\n\n📅 مدت: 30 روز\n📆 انقضا: $jalali\n👑 مالک: " . formatUserMention($owner_id),
        'parse_mode' => 'HTML',
        'reply_to_message_id' => $msg_id
    ]);
}