<?php
// admin.php — پنل مدیریت کامل و امن
session_start();

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/functions.php';
require_once __DIR__ . '/time.php';

// --- 1. سیستم لاگین و امنیت ---

// خروج از حساب
if (isset($_GET['logout'])) {
    session_destroy();
    header("Location: admin.php");
    exit;
}

$error_login = '';

// بررسی فرم ورود
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['login_pass'])) {
    // بررسی رمز عبور با آنچه در config.php تعریف شده
    if (defined('PANEL_PASSWORD') && $_POST['login_pass'] === PANEL_PASSWORD) {
        $_SESSION['admin_logged_in'] = true;
        header("Location: admin.php");
        exit;
    } else {
        $error_login = '❌ رمز عبور اشتباه است!';
    }
}

// اگر کاربر لاگین نکرده باشد، فرم ورود نمایش داده شود
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    ?>
    <!DOCTYPE html>
    <html lang="fa" dir="rtl">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>🔒 ورود به مدیریت</title>
        <style>
            body { background: #1e1e2e; color: #cdd6f4; font-family: Tahoma, sans-serif; display: flex; justify-content: center; align-items: center; height: 100vh; margin: 0; }
            .login-box { background: #181825; padding: 40px; border-radius: 12px; box-shadow: 0 4px 15px rgba(0,0,0,0.3); width: 320px; text-align: center; border: 1px solid #45475a; }
            h2 { margin-bottom: 20px; color: #89b4fa; }
            input { width: 100%; padding: 12px; margin: 15px 0; border-radius: 8px; border: 1px solid #45475a; background: #313244; color: white; box-sizing: border-box; text-align: center; font-size: 16px; }
            button { width: 100%; padding: 12px; background: #89b4fa; border: none; border-radius: 8px; color: #1e1e2e; font-weight: bold; cursor: pointer; transition: 0.3s; font-size: 16px; }
            button:hover { background: #b4befe; }
            .error { color: #f38ba8; margin-bottom: 15px; font-size: 14px; background: rgba(243, 139, 168, 0.1); padding: 10px; border-radius: 5px; }
        </style>
    </head>
    <body>
        <div class="login-box">
            <h2>🔐 پنل مدیریت ربات</h2>
            <?php if ($error_login): ?>
                <div class="error"><?= $error_login ?></div>
            <?php endif; ?>
            <form method="POST">
                <input type="password" name="login_pass" placeholder="رمز عبور را وارد کنید..." required autofocus>
                <button type="submit">ورود</button>
            </form>
        </div>
    </body>
    </html>
    <?php
    exit;
}

// --- 2. پردازش عملیات پنل (فقط برای کاربران لاگین شده) ---

$msg = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        // ایجاد کد
        if (isset($_POST['action']) && $_POST['action'] === 'create_code') {
            $days = max(1, (int)($_POST['days'] ?? 30));
            $code = strtoupper(substr(uniqid('VIP'), 0, 8) . '-' . strtoupper(bin2hex(random_bytes(3))));
            
            db()->prepare("INSERT INTO activation_codes (code, duration_days) VALUES (?, ?)")
                ->execute([$code, $days]);
                
            $msg = "✅ کد جدید ساخته شد: <b>$code</b> ($days روز)";
        } 
        // حذف کد
        elseif (isset($_POST['action']) && $_POST['action'] === 'delete_code' && isset($_POST['id'])) {
            db()->prepare("DELETE FROM activation_codes WHERE id = ?")->execute([$_POST['id']]);
            $msg = "🗑️ کد با موفقیت حذف شد.";
        } 
        // تمدید گروه
        elseif (isset($_POST['action']) && $_POST['action'] === 'extend_group' && isset($_POST['chat_id'])) {
            $days = max(1, (int)$_POST['days']);
            $chat_id = $_POST['chat_id'];
            
            $stmt = db()->prepare("SELECT expires_at FROM groups WHERE chat_id = ?");
            $stmt->execute([$chat_id]);
            $row = $stmt->fetch();
            
            if ($row) {
                // اگر گروه هنوز وقت دارد، از ادامه وقتش تمدید کن. اگر تمام شده، از الان تمدید کن.
                $current_exp = strtotime($row['expires_at']);
                $base_time = ($current_exp > time()) ? $current_exp : time();
                $new_exp = date('Y-m-d H:i:s', $base_time + ($days * 86400));
                
                db()->prepare("UPDATE groups SET expires_at = ? WHERE chat_id = ?")->execute([$new_exp, $chat_id]);
                $msg = "✅ گروه تمدید شد. اعتبار جدید تا: " . jdate('Y/m/d H:i', strtotime($new_exp));
            }
        } 
        // حذف گروه
        elseif (isset($_POST['action']) && $_POST['action'] === 'delete_group' && isset($_POST['chat_id'])) {
            $chat_id = $_POST['chat_id'];
            // حذف آبشاری اطلاعات
            db()->prepare("DELETE FROM groups WHERE chat_id = ?")->execute([$chat_id]);
            db()->prepare("DELETE FROM bot_admins WHERE chat_id = ?")->execute([$chat_id]);
            db()->prepare("DELETE FROM group_owners WHERE chat_id = ?")->execute([$chat_id]);
            db()->prepare("DELETE FROM vip_users WHERE chat_id = ?")->execute([$chat_id]);
            
            $msg = "🗑️ گروه $chat_id و تمام اطلاعات مربوطه حذف شد.";
        } 
        // پاکسازی منقضی‌ها
        elseif (isset($_POST['action']) && $_POST['action'] === 'cleanup_expired') {
            $count = db()->exec("DELETE FROM groups WHERE expires_at < NOW()");
            $msg = "🧹 تعداد $count گروه منقضی شده پاکسازی شد.";
        }
    } catch (Exception $e) {
        $msg = "❌ خطا: " . $e->getMessage();
    }
}
?>

<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>💎 پنل مدیریت ربات</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --bg-primary: #1e1e2e; --bg-secondary: #252536; --bg-tertiary: #313244;
            --text-primary: #cdd6f4; --text-secondary: #bac2de;
            --accent: #89b4fa; --accent-hover: #74a9f0; --success: #a6e3a1; --danger: #f38ba8;
            --card-bg: #181825; --border: #45475a;
        }
        [data-theme="light"] {
            --bg-primary: #f8f9fa; --bg-secondary: #e9ecef; --bg-tertiary: #dee2e6;
            --text-primary: #212529; --text-secondary: #495057; --accent: #3474eb;
            --card-bg: #ffffff; --border: #ced4da;
        }
        * { margin: 0; padding: 0; box-sizing: border-box; font-family: 'Segoe UI', Tahoma, sans-serif; }
        body { background: var(--bg-primary); color: var(--text-primary); min-height: 100vh; padding: 20px; }
        .container { max-width: 1200px; margin: 0 auto; }
        header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 30px; padding-bottom: 15px; border-bottom: 1px solid var(--border); }
        h1 { font-size: 24px; color: var(--accent); display: flex; gap: 10px; align-items: center; }
        .btn-logout { background: var(--danger); color: #fff; text-decoration: none; padding: 8px 15px; border-radius: 6px; font-size: 14px; font-weight: bold; transition: 0.2s; }
        .btn-logout:hover { opacity: 0.8; }
        .alert { padding: 15px; margin: 20px 0; border-radius: 8px; background: var(--bg-tertiary); border-left: 4px solid var(--accent); display: flex; align-items: center; gap: 10px; }
        .stats-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(240px, 1fr)); gap: 20px; margin: 30px 0; }
        .stat-card { background: var(--card-bg); padding: 20px; border-radius: 12px; border: 1px solid var(--border); box-shadow: 0 4px 6px rgba(0,0,0,0.05); }
        .stat-value { font-size: 32px; font-weight: 700; margin: 10px 0; color: var(--accent); }
        .section { background: var(--card-bg); border-radius: 16px; padding: 25px; margin: 30px 0; border: 1px solid var(--border); }
        .section-header { margin-bottom: 20px; font-size: 18px; font-weight: bold; border-bottom: 1px solid var(--border); padding-bottom: 10px; display: flex; align-items: center; gap: 10px; }
        form { display: flex; gap: 12px; flex-wrap: wrap; margin-bottom: 10px; }
        input, select, button { padding: 10px 16px; border-radius: 8px; border: 1px solid var(--border); background: var(--bg-tertiary); color: var(--text-primary); font-size: 14px; }
        [data-theme="light"] input, [data-theme="light"] select { background: white; }
        button { background: var(--accent); color: #111; font-weight: 600; cursor: pointer; border: none; transition: 0.2s; }
        button:hover { background: var(--accent-hover); }
        .btn-danger { background: var(--danger) !important; color: white !important; }
        .btn-success { background: var(--success) !important; color: #111 !important; }
        table { width: 100%; border-collapse: collapse; margin-top: 15px; }
        th, td { padding: 12px; text-align: right; border-bottom: 1px solid var(--border); }
        th { color: var(--text-secondary); font-size: 14px; }
        tr:hover { background: var(--bg-tertiary); }
        @media(max-width: 768px) { .stats-grid { grid-template-columns: 1fr; } }
    </style>
</head>
<body>
    <div class="container">
        <header>
            <h1><i class="fas fa-robot"></i> پنل مدیریت</h1>
            <div style="display:flex; gap:10px;">
                <button id="themeToggle" style="width:40px;">🌓</button>
                <a href="?logout=1" class="btn-logout"><i class="fas fa-sign-out-alt"></i> خروج</a>
            </div>
        </header>

        <?php if (!empty($msg)): ?>
        <div class="alert">
            <i class="fas fa-info-circle"></i> <span><?= $msg ?></span>
        </div>
        <?php endif; ?>

        <?php
        // دریافت آمار
        $active_groups = db()->query("SELECT COUNT(*) FROM groups WHERE expires_at > NOW()")->fetchColumn();
        $total_groups = db()->query("SELECT COUNT(*) FROM groups")->fetchColumn();
        $used_codes = db()->query("SELECT COUNT(*) FROM activation_codes WHERE used_by_chat_id IS NOT NULL")->fetchColumn();
        $total_codes = db()->query("SELECT COUNT(*) FROM activation_codes")->fetchColumn();
        $expired_groups = db()->query("SELECT COUNT(*) FROM groups WHERE expires_at < NOW()")->fetchColumn();
        ?>

        <div class="stats-grid">
            <div class="stat-card">
                <div><i class="fas fa-users"></i> گروه‌های فعال</div>
                <div class="stat-value"><?= $active_groups ?></div>
                <div style="font-size:12px; opacity:0.7;">از کل <?= $total_groups ?> گروه</div>
            </div>
            <div class="stat-card">
                <div><i class="fas fa-barcode"></i> کدهای مصرفی</div>
                <div class="stat-value"><?= $used_codes ?></div>
                <div style="font-size:12px; opacity:0.7;">از کل <?= $total_codes ?> کد</div>
            </div>
            <div class="stat-card">
                <div><i class="fas fa-exclamation-triangle"></i> منقضی شده</div>
                <div class="stat-value" style="color:var(--danger)"><?= $expired_groups ?></div>
                <div style="font-size:12px; opacity:0.7;">نیاز به بررسی</div>
            </div>
        </div>

        <!-- 1. ایجاد کد -->
        <div class="section">
            <div class="section-header"><i class="fas fa-plus-circle"></i> ایجاد کد فعال‌سازی</div>
            <form method="POST">
                <input type="hidden" name="action" value="create_code">
                <select name="days">
                    <option value="1">1 روز (تست)</option>
                    <option value="7">7 روز (هفتگی)</option>
                    <option value="30" selected>30 روز (ماهانه)</option>
                    <option value="90">90 روز (سه ماهه)</option>
                    <option value="365">1 سال</option>
                </select>
                <button type="submit"><i class="fas fa-magic"></i> ساخت کد</button>
            </form>
        </div>

        <!-- 2. پاکسازی منقضی‌ها -->
        <?php if($expired_groups > 0): ?>
        <div class="section" style="border: 1px solid var(--danger);">
            <div class="section-header" style="color:var(--danger); border-bottom-color:var(--danger);">
                <i class="fas fa-trash-alt"></i> پاکسازی انبوه
            </div>
            <p style="margin-bottom:15px;">⚠️ تعداد <b><?= $expired_groups ?></b> گروه منقضی شده در دیتابیس وجود دارد.</p>
            <form method="POST">
                <input type="hidden" name="action" value="cleanup_expired">
                <button type="submit" class="btn-danger" onclick="return confirm('آیا مطمئن هستید؟ این عملیات غیرقابل بازگشت است.')">
                    حذف تمام گروه‌های منقضی
                </button>
            </form>
        </div>
        <?php endif; ?>

        <!-- 3. مدیریت گروه‌ها -->
        <div class="section">
            <div class="section-header">
                <i class="fas fa-list"></i> لیست گروه‌ها
                <form method="GET" style="margin:0; margin-right:auto;">
                    <input type="text" name="search" placeholder="جستجو..." value="<?= htmlspecialchars($_GET['search'] ?? '') ?>">
                    <button type="submit"><i class="fas fa-search"></i></button>
                    <?php if(isset($_GET['search'])): ?>
                        <a href="admin.php" style="padding:10px; color:var(--text-secondary); text-decoration:none;">لغو</a>
                    <?php endif; ?>
                </form>
            </div>
            
            <?php
            $sql = "SELECT * FROM groups";
            $params = [];
            if (!empty($_GET['search'])) {
                $term = "%" . $_GET['search'] . "%";
                $sql .= " WHERE chat_id LIKE ? OR owner_id LIKE ?";
                $params = [$term, $term];
            }
            $sql .= " ORDER BY expires_at DESC LIMIT 50";
            $stmt = db()->prepare($sql);
            $stmt->execute($params);
            $rows = $stmt->fetchAll();
            ?>
            
            <div style="overflow-x:auto;">
            <table>
                <thead>
                    <tr>
                        <th>شناسه گروه</th>
                        <th>مالک</th>
                        <th>انقضا</th>
                        <th>وضعیت</th>
                        <th>عملیات</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($rows as $g): ?>
                    <tr>
                        <td><code><?= $g['chat_id'] ?></code></td>
                        <td><?= $g['owner_id'] ?></td>
                        <td dir="ltr"><?= jdate('Y/m/d H:i', strtotime($g['expires_at'])) ?></td>
                        <td>
                            <?php if (strtotime($g['expires_at']) > time()): ?>
                                <span style="color:var(--success);">فعال</span>
                            <?php else: ?>
                                <span style="color:var(--danger);">منقضی</span>
                            <?php endif; ?>
                        </td>
                        <td style="display:flex; gap:5px; justify-content:flex-end;">
                            <!-- تمدید -->
                            <form method="POST">
                                <input type="hidden" name="action" value="extend_group">
                                <input type="hidden" name="chat_id" value="<?= $g['chat_id'] ?>">
                                <input type="hidden" name="days" value="30">
                                <button type="submit" class="btn-success" title="تمدید 30 روز"><i class="fas fa-plus"></i></button>
                            </form>
                            <!-- حذف -->
                            <form method="POST" onsubmit="return confirm('آیا از حذف این گروه و تمام تنظیمات آن اطمینان دارید؟');">
                                <input type="hidden" name="action" value="delete_group">
                                <input type="hidden" name="chat_id" value="<?= $g['chat_id'] ?>">
                                <button type="submit" class="btn-danger" title="حذف گروه"><i class="fas fa-trash"></i></button>
                            </form>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            </div>
        </div>

        <!-- 4. کدهای فعال -->
        <div class="section">
            <div class="section-header"><i class="fas fa-key"></i> کدهای استفاده نشده (20 تای آخر)</div>
            <?php
            $codes = db()->query("SELECT * FROM activation_codes WHERE used_by_chat_id IS NULL ORDER BY id DESC LIMIT 20")->fetchAll();
            ?>
            <div style="overflow-x:auto;">
            <table>
                <thead>
                    <tr>
                        <th>کد</th>
                        <th>اعتبار</th>
                        <th>تاریخ ساخت</th>
                        <th>حذف</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($codes as $c): ?>
                    <tr>
                        <td><code style="font-size:16px; user-select:all; cursor:pointer;"><?= $c['code'] ?></code></td>
                        <td><?= $c['duration_days'] ?> روز</td>
                        <td><?= jdate('Y/m/d', strtotime($c['created_at'])) ?></td>
                        <td>
                            <form method="POST" onsubmit="return confirm('حذف کد؟');">
                                <input type="hidden" name="action" value="delete_code">
                                <input type="hidden" name="id" value="<?= $c['id'] ?>">
                                <button type="submit" class="btn-danger"><i class="fas fa-times"></i></button>
                            </form>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            </div>
        </div>

    </div>

    <script>
        const toggle = document.getElementById('themeToggle');
        const currentTheme = localStorage.getItem('theme') || 'dark';
        document.documentElement.setAttribute('data-theme', currentTheme);
        
        toggle.addEventListener('click', () => {
            const theme = document.documentElement.getAttribute('data-theme') === 'dark' ? 'light' : 'dark';
            document.documentElement.setAttribute('data-theme', theme);
            localStorage.setItem('theme', theme);
        });
    </script>
</body>
</html>