<?php
/**
 * سیستم Mia - مدیریت قفل‌های گروه
 * MIA Lock Management System
 * نسخه اصلاح شده: محدودیت دسترسی به دکمه‌ها
 */

require_once 'config.php';

/**
 * ایجاد جداول Mia در دیتابیس
 */
function createMiaTables() {
    $db = Database::instance()->pdo();
    
    // جدول قفل‌ها
    $db->exec("CREATE TABLE IF NOT EXISTS mia_locks (
        id INT AUTO_INCREMENT PRIMARY KEY,
        group_id BIGINT NOT NULL,
        lock_type VARCHAR(50) NOT NULL,
        is_active TINYINT(1) DEFAULT 0,
        expire_time DATETIME DEFAULT NULL,
        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        UNIQUE KEY unique_lock (group_id, lock_type),
        INDEX idx_group_id (group_id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci");
}

createMiaTables();

/**
 * کلاس اصلی Mia
 */
class MiaSystem {
    private $db;
    private $message;
    private $chat_id;
    private $from_id;
    private $message_id;
    private $text;
    
    // لیست تمام قفل‌ها
    private $lock_types = [
        'link' => 'لینک',
        'forward' => 'فوروارد',
        'tag' => 'تگ',
        'mention' => 'منشن',
        'hashtag' => 'هشتگ',
        'username' => 'یوزرنیم',
        'sticker' => 'استیکر',
        'gif' => 'گیف',
        'photo' => 'عکس',
        'video' => 'ویدیو',
        'voice' => 'ویس',
        'audio' => 'آهنگ',
        'file' => 'فایل',
        'location' => 'موقعیت',
        'contact' => 'مخاطب',
        'game' => 'بازی',
        'bot' => 'ربات',
        'inline' => 'اینلاین',
        'emoji' => 'ایموجی',
        'farsi' => 'فارسی',
        'english' => 'انگلیسی',
        'arabic' => 'عربی',
        'longtext' => 'پیام بلند',
        'service' => 'سرویس',
        'spoiler' => 'اسپویلر',
        'porn' => 'پورن'
    ];
    
    public function __construct($message) {
        $this->db = Database::instance()->pdo();
        $this->message = $message;
        $this->chat_id = $message['chat']['id'];
        $this->from_id = $message['from']['id'];
        $this->message_id = $message['message_id'];
        $this->text = trim($message['text'] ?? '');
    }
    
    /**
     * پردازش دستورات
     */
    public function processCommand() {
        if (!$this->isGroupActive()) {
            return false;
        }
        
        if (!$this->hasPermission()) {
            return false;
        }
        
        $text = $this->text;
        
        // پنل قفل‌ها
        if ($text == 'قفل‌ها' || $text == 'قفلها' || $text == 'پنل قفل') {
            return $this->showLocksPanel();
        }
        
        // قفل کردن تکی
        foreach ($this->lock_types as $type => $name) {
            if ($text == "قفل {$name}" || $text == "قفل کردن {$name}") {
                return $this->lockFeature($type);
            }
            if ($text == "باز کردن {$name}" || $text == "بازکردن {$name}") {
                return $this->unlockFeature($type);
            }
        }
        
        return false;
    }
    
    /**
     * بررسی پیام و اعمال قفل‌ها
     */
    public function checkMessage() {
        if (!$this->isGroupActive()) {
            return false;
        }
        
        // بررسی مقام کاربر - اگر مقام داشت، هیچ قفلی اعمال نمی‌شود
        if ($this->hasRank($this->from_id)) {
            return false;
        }
        
        $should_delete = false;
        $reason = '';
        
        // قفل سرویس (پیام‌های گزارشی تلگرام)
        if ($this->isServiceMessage() && $this->isLocked('service')) {
            deleteMessage($this->chat_id, $this->message_id);
            return true;
        }
        
        // قفل فوروارد
        if (isset($this->message['forward_date']) && $this->isLocked('forward')) {
            $should_delete = true;
            $reason = 'فوروارد';
        }
        
        // قفل ربات (اضافه کردن ربات)
        if (isset($this->message['new_chat_members'])) {
            foreach ($this->message['new_chat_members'] as $member) {
                if (isset($member['is_bot']) && $member['is_bot'] && $this->isLocked('bot')) {
                    $should_delete = true;
                    $reason = 'ربات';
                    break;
                }
            }
        }
        
        // قفل‌های محتوا
        if (isset($this->message['sticker']) && $this->isLocked('sticker')) {
            $should_delete = true;
            $reason = 'استیکر';
        }
        
        if (isset($this->message['animation']) && $this->isLocked('gif')) {
            $should_delete = true;
            $reason = 'گیف';
        }
        
        if (isset($this->message['photo']) && $this->isLocked('photo')) {
            $should_delete = true;
            $reason = 'عکس';
        }
        
        if (isset($this->message['video']) && $this->isLocked('video')) {
            $should_delete = true;
            $reason = 'ویدیو';
        }
        
        if (isset($this->message['voice']) && $this->isLocked('voice')) {
            $should_delete = true;
            $reason = 'ویس';
        }
        
        if (isset($this->message['audio']) && $this->isLocked('audio')) {
            $should_delete = true;
            $reason = 'آهنگ';
        }
        
        if (isset($this->message['document']) && $this->isLocked('file')) {
            $should_delete = true;
            $reason = 'فایل';
        }
        
        if (isset($this->message['location']) && $this->isLocked('location')) {
            $should_delete = true;
            $reason = 'موقعیت';
        }
        
        if (isset($this->message['contact']) && $this->isLocked('contact')) {
            $should_delete = true;
            $reason = 'مخاطب';
        }
        
        if (isset($this->message['game']) && $this->isLocked('game')) {
            $should_delete = true;
            $reason = 'بازی';
        }
        
        if (isset($this->message['via_bot']) && $this->isLocked('inline')) {
            $should_delete = true;
            $reason = 'اینلاین';
        }
        
        // بررسی محتوای متنی
        if (!empty($this->text)) {
            // قفل لینک
            if ($this->isLocked('link') && $this->hasLink($this->text)) {
                $should_delete = true;
                $reason = 'لینک';
            }
            
            // قفل تگ
            if (!$should_delete && $this->isLocked('tag') && preg_match('/#[\w\x{0600}-\x{06FF}]+/u', $this->text)) {
                $should_delete = true;
                $reason = 'تگ';
            }
            
            // قفل منشن
            if (!$should_delete && $this->isLocked('mention') && isset($this->message['entities'])) {
                foreach ($this->message['entities'] as $entity) {
                    if ($entity['type'] == 'text_mention' || $entity['type'] == 'mention') {
                        $should_delete = true;
                        $reason = 'منشن';
                        break;
                    }
                }
            }
            
            // قفل هشتگ
            if (!$should_delete && $this->isLocked('hashtag') && preg_match('/#[\w\x{0600}-\x{06FF}]+/u', $this->text)) {
                $should_delete = true;
                $reason = 'هشتگ';
            }
            
            // قفل یوزرنیم
            if (!$should_delete && $this->isLocked('username') && preg_match('/@\w+/', $this->text)) {
                $should_delete = true;
                $reason = 'یوزرنیم';
            }
            
            // قفل ایموجی
            if (!$should_delete && $this->isLocked('emoji') && preg_match('/[\x{1F600}-\x{1F64F}\x{1F300}-\x{1F5FF}\x{1F680}-\x{1F6FF}\x{2600}-\x{26FF}\x{2700}-\x{27BF}]/u', $this->text)) {
                $should_delete = true;
                $reason = 'ایموجی';
            }
            
            // قفل فارسی
            if (!$should_delete && $this->isLocked('farsi') && preg_match('/[\x{0600}-\x{06FF}]/u', $this->text)) {
                $should_delete = true;
                $reason = 'فارسی';
            }
            
            // قفل انگلیسی
            if (!$should_delete && $this->isLocked('english') && preg_match('/[a-zA-Z]/', $this->text)) {
                $should_delete = true;
                $reason = 'انگلیسی';
            }
            
            // قفل عربی
            if (!$should_delete && $this->isLocked('arabic') && preg_match('/[\x{0600}-\x{06FF}]/u', $this->text) && !preg_match('/[پچژگ]/u', $this->text)) {
                $should_delete = true;
                $reason = 'عربی';
            }
            
            // قفل پیام بلند (بیش از 500 کاراکتر)
            if (!$should_delete && $this->isLocked('longtext') && mb_strlen($this->text) > 500) {
                $should_delete = true;
                $reason = 'پیام بلند';
            }
            
            // قفل اسپویلر
            if (!$should_delete && $this->isLocked('spoiler') && isset($this->message['entities'])) {
                foreach ($this->message['entities'] as $entity) {
                    if ($entity['type'] == 'spoiler') {
                        $should_delete = true;
                        $reason = 'اسپویلر';
                        break;
                    }
                }
            }
        }
        
        // حذف پیام
        if ($should_delete) {
            deleteMessage($this->chat_id, $this->message_id);
            return true;
        }
        
        return false;
    }
    
    /**
     * نمایش پنل قفل‌ها
     * ✅ اصلاح شده: افزودن ID کاربر به دکمه‌ها
     */
    private function showLocksPanel() {
        $locks = $this->getAllLocks();
        
        $text = "🔐 <b>پنل مدیریت قفل‌های گروه</b>\n\n";
        $text .= "📊 <b>وضعیت قفل‌ها:</b>\n\n";
        
        foreach ($this->lock_types as $type => $name) {
            $status = isset($locks[$type]) && $locks[$type] ? '✅' : '❌';
            $text .= "• {$name}: {$status}\n";
        }
        
        $text .= "\n💡 <b>راهنما:</b>\n";
        $text .= "• ✅ = فعال\n";
        $text .= "• ❌ = غیرفعال\n\n";
        $text .= "⚡️ <b>توجه:</b> این پنل فقط برای شما کار می‌کند.";
        
        // ساخت کیبورد با شناسه کاربر
        $keyboard = ['inline_keyboard' => []];
        $row = [];
        $count = 0;
        
        // اضافه کردن شناسه کاربر به دکمه‌ها برای امنیت
        $userId = $this->from_id;
        
        foreach ($this->lock_types as $type => $name) {
            $status = isset($locks[$type]) && $locks[$type] ? '✅' : '❌';
            $row[] = [
                'text' => "{$status} {$name}",
                'callback_data' => "mia_toggle_{$type}_{$userId}" // شناسه کاربر اضافه شد
            ];
            $count++;
            
            if ($count % 2 == 0) {
                $keyboard['inline_keyboard'][] = $row;
                $row = [];
            }
        }
        
        if (!empty($row)) {
            $keyboard['inline_keyboard'][] = $row;
        }
        
        $keyboard['inline_keyboard'][] = [
            ['text' => '❌ بستن پنل', 'callback_data' => "mia_close_{$userId}"] // شناسه کاربر اضافه شد
        ];
        
        sendMessage($this->chat_id, $text, $keyboard, $this->message_id);
        return true;
    }
    
    /**
     * قفل کردن ویژگی (تکی)
     */
    private function lockFeature($type) {
        if ($this->isLocked($type)) {
            $name = $this->lock_types[$type];
            $this->reply("ℹ️ قفل {$name} در حال حاضر فعال است.");
            return true;
        }
        
        $this->setLock($type, true);
        $name = $this->lock_types[$type];
        $this->reply("✅ قفل {$name} با موفقیت فعال شد.");
        return true;
    }
    
    /**
     * باز کردن قفل (تکی)
     */
    private function unlockFeature($type) {
        if (!$this->isLocked($type)) {
            $name = $this->lock_types[$type];
            $this->reply("ℹ️ قفل {$name} در حال حاضر غیرفعال است.");
            return true;
        }
        
        $this->setLock($type, false);
        $name = $this->lock_types[$type];
        $this->reply("✅ قفل {$name} با موفقیت غیرفعال شد.");
        return true;
    }
    
    /**
     * مدیریت callback های پنل
     * ✅ اصلاح شده: بررسی شناسه کاربر
     */
    public function handleCallback($callback_data, $callback_id, $message_id, $clicker_id) {
        // استخراج شناسه کاربر صاحب پنل از callback_data
        // فرمت: mia_toggle_TYPE_USERID یا mia_close_USERID
        
        if (strpos($callback_data, 'mia_toggle_') === 0) {
            // جدا کردن بخش‌های دیتا
            $parts = explode('_', $callback_data);
            // آخرین بخش همیشه شناسه کاربر است
            $owner_id = end($parts);
            // نوع قفل، بخش سوم (ایندکس 2) است یا اگر نام قفل خط تیره داشته باشد باید مدیریت شود
            // ساده‌ترین راه: حذف mia_toggle_ و حذف _USERID از انتها
            $type_part = str_replace('mia_toggle_', '', $callback_data);
            $type = str_replace('_' . $owner_id, '', $type_part);
            
            // بررسی دسترسی: آیا کلیک کننده همان صاحب پنل است؟
            if ($clicker_id != $owner_id) {
                sendRequest('answerCallbackQuery', [
                    'callback_query_id' => $callback_id,
                    'text' => '⛔️ این پنل برای شما نیست! لطفا دستور "قفل‌ها" را خودتان ارسال کنید.',
                    'show_alert' => true
                ]);
                return true;
            }
            
            if (isset($this->lock_types[$type])) {
                $current = $this->isLocked($type);
                $this->setLock($type, !$current);
                
                // بروزرسانی پنل (ارسال owner_id برای بازسازی دکمه‌ها)
                $this->updatePanel($message_id, $owner_id);
                
                $name = $this->lock_types[$type];
                $status = !$current ? 'فعال' : 'غیرفعال';
                
                sendRequest('answerCallbackQuery', [
                    'callback_query_id' => $callback_id,
                    'text' => "✅ قفل {$name} {$status} شد"
                ]);
            }
            return true;
        }
        
        if (strpos($callback_data, 'mia_close_') === 0) {
            $parts = explode('_', $callback_data);
            $owner_id = end($parts);
            
            if ($clicker_id != $owner_id) {
                sendRequest('answerCallbackQuery', [
                    'callback_query_id' => $callback_id,
                    'text' => '⛔️ این پنل برای شما نیست!',
                    'show_alert' => true
                ]);
                return true;
            }
            
            editMessage($this->chat_id, $message_id, "✅ <b>پنل بسته شد</b>");
            sendRequest('answerCallbackQuery', [
                'callback_query_id' => $callback_id,
                'text' => 'پنل بسته شد'
            ]);
            return true;
        }
        
        return false;
    }
    
    /**
     * بروزرسانی پنل
     * ✅ اصلاح شده: دریافت owner_id برای حفظ امنیت دکمه‌ها
     */
    private function updatePanel($message_id, $owner_id) {
        $locks = $this->getAllLocks();
        
        $text = "🔐 <b>پنل مدیریت قفل‌های گروه</b>\n\n";
        $text .= "📊 <b>وضعیت قفل‌ها:</b>\n\n";
        
        foreach ($this->lock_types as $type => $name) {
            $status = isset($locks[$type]) && $locks[$type] ? '✅' : '❌';
            $text .= "• {$name}: {$status}\n";
        }
        
        $text .= "\n💡 <b>راهنما:</b>\n";
        $text .= "• ✅ = فعال\n";
        $text .= "• ❌ = غیرفعال\n\n";
        $text .= "⚡️ <b>توجه:</b> این پنل فقط برای شما کار می‌کند.";
        
        $keyboard = ['inline_keyboard' => []];
        $row = [];
        $count = 0;
        
        foreach ($this->lock_types as $type => $name) {
            $status = isset($locks[$type]) && $locks[$type] ? '✅' : '❌';
            $row[] = [
                'text' => "{$status} {$name}",
                'callback_data' => "mia_toggle_{$type}_{$owner_id}" // شناسه کاربر حفظ شد
            ];
            $count++;
            
            if ($count % 2 == 0) {
                $keyboard['inline_keyboard'][] = $row;
                $row = [];
            }
        }
        
        if (!empty($row)) {
            $keyboard['inline_keyboard'][] = $row;
        }
        
        $keyboard['inline_keyboard'][] = [
            ['text' => '❌ بستن پنل', 'callback_data' => "mia_close_{$owner_id}"] // شناسه کاربر حفظ شد
        ];
        
        editMessage($this->chat_id, $message_id, $text, $keyboard);
    }
    
    // توابع کمکی
    
    private function isGroupActive() {
        $stmt = $this->db->prepare("SELECT is_active, expire_date FROM groups_data WHERE group_id = ?");
        $stmt->execute([$this->chat_id]);
        $result = $stmt->fetch();
        
        if (!$result || $result['is_active'] != 1) {
            return false;
        }
        
        if ($result['expire_date'] && strtotime($result['expire_date']) < time()) {
            return false;
        }
        
        return true;
    }
    
    private function hasPermission() {
        if ($this->from_id == SUDO_ID) return true;
        if ($this->isCreator($this->from_id)) return true;
        if ($this->isOwner($this->from_id)) return true;
        if ($this->isDeputy($this->from_id)) return true;
        if ($this->isAdvisor($this->from_id)) return true;
        return false;
    }
    
    private function hasRank($user_id) {
        if ($user_id == SUDO_ID) return true;
        if ($this->isCreator($user_id)) return true;
        if ($this->isOwner($user_id)) return true;
        if ($this->isDeputy($user_id)) return true;
        if ($this->isAdvisor($user_id)) return true;
        if ($this->isAdmin($user_id)) return true;
        if ($this->isVip($user_id)) return true;
        return false;
    }
    
    private function isCreator($user_id) {
        $stmt = $this->db->prepare("SELECT * FROM group_admins WHERE group_id = ? AND user_id = ? AND is_creator = 1");
        $stmt->execute([$this->chat_id, $user_id]);
        return $stmt->fetch() !== false;
    }
    
    private function isOwner($user_id) {
        $stmt = $this->db->prepare("SELECT * FROM mir_owners WHERE group_id = ? AND user_id = ?");
        $stmt->execute([$this->chat_id, $user_id]);
        return $stmt->fetch() !== false;
    }
    
    private function isDeputy($user_id) {
        $stmt = $this->db->prepare("SELECT * FROM mir_deputies WHERE group_id = ? AND user_id = ?");
        $stmt->execute([$this->chat_id, $user_id]);
        return $stmt->fetch() !== false;
    }
    
    private function isAdvisor($user_id) {
        $stmt = $this->db->prepare("SELECT * FROM mir_advisors WHERE group_id = ? AND user_id = ?");
        $stmt->execute([$this->chat_id, $user_id]);
        return $stmt->fetch() !== false;
    }
    
    private function isAdmin($user_id) {
        $stmt = $this->db->prepare("SELECT * FROM mir_admins WHERE group_id = ? AND user_id = ?");
        $stmt->execute([$this->chat_id, $user_id]);
        return $stmt->fetch() !== false;
    }
    
    private function isVip($user_id) {
        $stmt = $this->db->prepare("SELECT * FROM mir_vips WHERE group_id = ? AND user_id = ?");
        $stmt->execute([$this->chat_id, $user_id]);
        return $stmt->fetch() !== false;
    }
    
    private function isLocked($type) {
        $stmt = $this->db->prepare("SELECT is_active FROM mia_locks WHERE group_id = ? AND lock_type = ?");
        $stmt->execute([$this->chat_id, $type]);
        $result = $stmt->fetch();
        return $result && $result['is_active'] == 1;
    }
    
    private function setLock($type, $active) {
        $stmt = $this->db->prepare("INSERT INTO mia_locks (group_id, lock_type, is_active) VALUES (?, ?, ?) ON DUPLICATE KEY UPDATE is_active = ?");
        $stmt->execute([$this->chat_id, $type, $active ? 1 : 0, $active ? 1 : 0]);
    }
    
    private function getAllLocks() {
        $stmt = $this->db->prepare("SELECT lock_type, is_active FROM mia_locks WHERE group_id = ?");
        $stmt->execute([$this->chat_id]);
        $rows = $stmt->fetchAll();
        
        $locks = [];
        foreach ($rows as $row) {
            $locks[$row['lock_type']] = $row['is_active'] == 1;
        }
        
        return $locks;
    }
    
    private function hasLink($text) {
        return preg_match('/(https?:\/\/[^\s]+|t\.me\/[^\s]+|@[^\s]+\.com)/i', $text);
    }
    
    private function isServiceMessage() {
        return isset($this->message['new_chat_members']) ||
               isset($this->message['left_chat_member']) ||
               isset($this->message['new_chat_title']) ||
               isset($this->message['new_chat_photo']) ||
               isset($this->message['delete_chat_photo']) ||
               isset($this->message['group_chat_created']) ||
               isset($this->message['supergroup_chat_created']) ||
               isset($this->message['channel_chat_created']) ||
               isset($this->message['migrate_to_chat_id']) ||
               isset($this->message['migrate_from_chat_id']) ||
               isset($this->message['pinned_message']);
    }
    
    private function reply($text) {
        sendMessage($this->chat_id, $text, null, $this->message_id);
    }
}

/**
 * پردازش دستورات Mia
 */
function processMiaCommand($message) {
    $mia = new MiaSystem($message);
    return $mia->processCommand();
}

/**
 * بررسی پیام با Mia
 */
function checkMiaMessage($message) {
    $mia = new MiaSystem($message);
    return $mia->checkMessage();
}

/**
 * مدیریت callback های Mia
 * ✅ اصلاح شده: ارسال شناسه کلیک کننده
 */
function handleMiaCallback($callback) {
    $data = $callback['data'];
    
    if (strpos($data, 'mia_') === 0) {
        $message = $callback['message'];
        $mia = new MiaSystem($message);
        
        // ارسال شناسه کلیک کننده برای بررسی امنیتی
        $mia->handleCallback(
            $data, 
            $callback['id'], 
            $message['message_id'],
            $callback['from']['id'] // شناسه کلیک کننده
        );
        return true;
    }
    
    return false;
}

?>