<?php
/**
 * سیستم ME - قابلیت‌های کاربری و اجتماعی
 * ME System - User Features and Social Functions
 * نسخه نهایی - اصلاح شده (محدودیت دستور آیدی برای ویژه به بالا)
 */

require_once 'config.php';
require_once 'jdate.php';

// اطمینان از وجود فایل توابع
if (file_exists('functions.php')) {
    require_once 'functions.php';
}

/**
 * ایجاد جداول ME در دیتابیس
 */
function createMeTables() {
    $db = Database::instance()->pdo();
    
    // جدول اطلاعات کاربران
    $db->exec("CREATE TABLE IF NOT EXISTS me_users (
        id INT AUTO_INCREMENT PRIMARY KEY,
        group_id BIGINT NOT NULL,
        user_id BIGINT NOT NULL,
        first_name VARCHAR(255),
        username VARCHAR(255),
        nickname VARCHAR(255),
        origin VARCHAR(255),
        daily_messages INT DEFAULT 0,
        total_messages INT DEFAULT 0,
        last_message_date DATE,
        last_seen DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        UNIQUE KEY unique_user (group_id, user_id),
        INDEX idx_group_user (group_id, user_id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci");
    
    // جدول پنل‌های باز (برای بستن خودکار)
    $db->exec("CREATE TABLE IF NOT EXISTS me_open_panels (
        id INT AUTO_INCREMENT PRIMARY KEY,
        group_id BIGINT NOT NULL,
        message_id BIGINT NOT NULL,
        panel_type VARCHAR(50) NOT NULL,
        opened_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_opened_at (opened_at)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci");
}

createMeTables();

/**
 * کلاس اصلی ME
 */
class MeSystem {
    private $db;
    private $message;
    private $chat_id;
    private $from_id;
    private $message_id;
    private $text;
    
    public function __construct($message) {
        $this->db = Database::instance()->pdo();
        
        $this->message = $message;
        $this->chat_id = $message['chat']['id'];
        $this->from_id = $message['from']['id'];
        $this->message_id = $message['message_id'];
        $this->text = trim($message['text'] ?? '');
        
        // به‌روزرسانی آمار پیام
        $this->updateMessageStats($message['from']);
    }
    
    /**
     * پردازش دستورات ME
     */
    public function process() {
        if (!$this->isGroupActive()) {
            return false;
        }
        
        $text = $this->text;
        
        // آیدی
        if ($text == 'آیدی') {
            return $this->showUserInfo();
        }
        
        // لقب
        if ($text == 'لقب من') {
            return $this->showNickname();
        }
        if ($text == 'تنظیم لقب' || preg_match('/^تنظیم لقب (.+)$/u', $text)) {
            return $this->setNickname();
        }
        if ($text == 'حذف لقب') {
            return $this->removeNickname();
        }
        
        // اصل
        if ($text == 'اصل من') {
            return $this->showOrigin();
        }
        if ($text == 'تنظیم اصل' || preg_match('/^تنظیم اصل (.+)$/u', $text)) {
            return $this->setOrigin();
        }
        if ($text == 'حذف اصل') {
            return $this->removeOrigin();
        }
        
        // تگ
        if ($text == 'تگ') {
            return $this->showTagPanel();
        }
        
        // بیو
        if ($text == 'بیو') {
            return $this->showRandomBio();
        }
        if (preg_match('/^بیو (.+)$/u', $text)) {
            return $this->showBioWithKeyword();
        }
        
        // فونت
        if (preg_match('/^فونت (.+)$/u', $text)) {
            return $this->showFonts();
        }
        
        // لینک
        if ($text == 'لینک') {
            return $this->showGroupLink();
        }
        
        // ساعت
        if ($text == 'ساعت') {
            return $this->showTime();
        }
        
        // سنجاق
        if ($text == 'سنجاق' || $text == 'Pin') {
            return $this->pinMessage();
        }
        if ($text == 'حذف سنجاق' || $text == 'Unpin') {
            return $this->unpinMessage();
        }
        
        // تنظیم عکس
        if ($text == 'تنظیم عکس' || $text == 'SetPhoto') {
            return $this->setGroupPhoto();
        }
        if ($text == 'حذف عکس') {
            return $this->deleteGroupPhoto();
        }
        
        // حذف پیام
        if ($text == 'حذف') {
            return $this->deleteMessage();
        }
        if (preg_match('/^حذف (\d+)$/u', $text)) {
            return $this->deleteMultipleMessages();
        }
        if (preg_match('/^حذف (\d+) روز$/u', $text)) {
            return $this->deleteMessagesByDate();
        }
        
        return false;
    }
    
    /**
     * نمایش اطلاعات کاربر با عکس
     * (محدود شده برای ویژه و بالاتر)
     */
    private function showUserInfo() {
        if (!$this->hasAccess('id')) {
            // اختیاری: ارسال پیام عدم دسترسی
            $this->reply("⛔️ این دستور فقط مخصوص مقام‌داران و کاربران ویژه است.");
            return true;
        }
        
        $target = $this->getTargetUser();
        $user_data = $this->getUserData($target['id']);
        $rank = $this->getUserRank($target['id']);
        
        $nickname = $user_data['nickname'] ?? 'ندارد';
        $origin = $user_data['origin'] ?? 'ندارد';
        $daily = $user_data['daily_messages'] ?? 0;
        $total = $user_data['total_messages'] ?? 0;
        
        $text = "👤 <b>اطلاعات کاربری</b>\n\n";
        $text .= "👤 نام: " . $this->getMention($target) . "\n";
        $text .= "🆔 یوزرنیم: " . (isset($target['username']) ? "@{$target['username']}" : "ندارد") . "\n";
        $text .= "🔢 شناسه: <code>{$target['id']}</code>\n";
        $text .= "🏷 لقب: {$nickname}\n";
        $text .= "🌍 اصل: {$origin}\n";
        $text .= "👑 مقام: {$rank}\n";
        $text .= "📊 چت‌های امروز: {$daily}\n";
        $text .= "📈 کل چت‌ها: {$total}";
        
        // استفاده از تابع global اگر وجود دارد، در غیر این صورت متد داخلی
        if (function_exists('sendRequest')) {
            $photos = sendRequest('getUserProfilePhotos', [
                'user_id' => $target['id'],
                'limit' => 1
            ]);
        } else {
            $this->reply($text);
            return true;
        }
        
        if ($photos && isset($photos['result']['photos'][0][0]['file_id'])) {
            $file_id = $photos['result']['photos'][0][0]['file_id'];
            sendRequest('sendPhoto', [
                'chat_id' => $this->chat_id,
                'photo' => $file_id,
                'caption' => $text,
                'parse_mode' => 'HTML',
                'reply_to_message_id' => $this->message_id
            ]);
        } else {
            $this->reply($text);
        }
        
        return true;
    }
    
    /**
     * تنظیم لقب
     */
    private function setNickname() {
        if (!$this->hasAccess('nickname')) return false;
        
        preg_match('/^تنظیم لقب (.+)$/u', $this->text, $matches);
        $nickname = $matches[1] ?? null;
        
        if (!$nickname && !isset($this->message['reply_to_message'])) {
            $this->reply("❌ لطفاً لقب را وارد کنید یا روی پیام کاربر ریپلای کنید.");
            return true;
        }
        
        $target = isset($this->message['reply_to_message']) ? 
            $this->message['reply_to_message']['from'] : 
            $this->message['from'];
        
        if (!$nickname) {
            $this->reply("❌ لطفاً لقب را بعد از دستور وارد کنید.\nمثال: تنظیم لقب پادشاه");
            return true;
        }
        
        if (mb_strlen($nickname) > 50) {
            $this->reply("❌ لقب نباید بیشتر از 50 کاراکتر باشد!");
            return true;
        }
        
        $this->updateUserField($target['id'], 'nickname', $nickname);
        
        $mention = $this->getMention($target);
        $this->reply("✅ لقب کاربر {$mention}\n›› به «{$nickname}» تنظیم شد.");
        
        return true;
    }
    
    /**
     * نمایش لقب
     */
    private function showNickname() {
        if (!$this->hasAccess('nickname')) return false;
        
        $target = $this->getTargetUser();
        $user_data = $this->getUserData($target['id']);
        
        if (!$user_data || !$user_data['nickname']) {
            $mention = $this->getMention($target);
            $this->reply("⚠️ کاربر {$mention}\n›› لقبی ندارد.");
            return true;
        }
        
        $mention = $this->getMention($target);
        $this->reply("🏷 لقب کاربر {$mention}\n›› {$user_data['nickname']}");
        
        return true;
    }
    
    /**
     * حذف لقب
     */
    private function removeNickname() {
        if (!$this->hasAccess('nickname')) return false;
        
        $target = $this->getTargetUser();
        $this->updateUserField($target['id'], 'nickname', null);
        
        $mention = $this->getMention($target);
        $this->reply("✅ لقب کاربر {$mention}\n›› حذف شد.");
        
        return true;
    }
    
    /**
     * تنظیم اصل
     */
    private function setOrigin() {
        if (!$this->hasAccess('origin')) return false;
        
        preg_match('/^تنظیم اصل (.+)$/u', $this->text, $matches);
        $origin = $matches[1] ?? null;
        
        if (!$origin && !isset($this->message['reply_to_message'])) {
            $this->reply("❌ لطفاً اصل را وارد کنید یا روی پیام کاربر ریپلای کنید.");
            return true;
        }
        
        $target = isset($this->message['reply_to_message']) ? 
            $this->message['reply_to_message']['from'] : 
            $this->message['from'];
        
        if (!$origin) {
            $this->reply("❌ لطفاً اصل را بعد از دستور وارد کنید.\nمثال: تنظیم اصل تهران");
            return true;
        }
        
        if (mb_strlen($origin) > 50) {
            $this->reply("❌ اصل نباید بیشتر از 50 کاراکتر باشد!");
            return true;
        }
        
        $this->updateUserField($target['id'], 'origin', $origin);
        
        $mention = $this->getMention($target);
        $this->reply("✅ اصل کاربر {$mention}\n›› به «{$origin}» تنظیم شد.");
        
        return true;
    }
    
    /**
     * نمایش اصل
     */
    private function showOrigin() {
        if (!$this->hasAccess('origin')) return false;
        
        $target = $this->getTargetUser();
        $user_data = $this->getUserData($target['id']);
        
        if (!$user_data || !$user_data['origin']) {
            $mention = $this->getMention($target);
            $this->reply("⚠️ کاربر {$mention}\n›› اصلی ندارد.");
            return true;
        }
        
        $mention = $this->getMention($target);
        $this->reply("🌍 اصل کاربر {$mention}\n›› {$user_data['origin']}");
        
        return true;
    }
    
    /**
     * حذف اصل
     */
    private function removeOrigin() {
        if (!$this->hasAccess('origin')) return false;
        
        $target = $this->getTargetUser();
        $this->updateUserField($target['id'], 'origin', null);
        
        $mention = $this->getMention($target);
        $this->reply("✅ اصل کاربر {$mention}\n›› حذف شد.");
        
        return true;
    }
    
    /**
     * نمایش پنل تگ
     */
    private function showTagPanel() {
        if (!$this->hasAccess('tag')) return false;
        
        // تشخیص پیام هدف برای تگ کردن
        // اگر روی پیامی ریپلای شده باشد، آیدی آن را می‌گیریم
        // در غیر این صورت، آیدی همین دستور "تگ" را می‌گیریم
        if (isset($this->message['reply_to_message'])) {
            $target_msg_id = $this->message['reply_to_message']['message_id'];
        } else {
            $target_msg_id = $this->message_id;
        }
        
        $keyboard = [
            'inline_keyboard' => [
                [
                    // ارسال آیدی پیام هدف در کال‌بک دیتا
                    ['text' => '👥 تگ اعضا', 'callback_data' => 'me_tag_members_' . $target_msg_id],
                    ['text' => '👨‍💼 تگ ادمین‌ها', 'callback_data' => 'me_tag_admins_' . $target_msg_id]
                ],
                [
                    ['text' => '⚡️ تگ فعال‌ها', 'callback_data' => 'me_tag_active_' . $target_msg_id]
                ],
                [
                    ['text' => '❌ بستن پنل', 'callback_data' => 'me_close_panel']
                ]
            ]
        ];
        
        $text = "🏷 <b>پنل تگ کاربران</b>\n\n";
        $text .= "لطفاً یکی از گزینه‌های زیر را انتخاب کنید:";
        
        // پنل روی دستور کاربر ریپلای می‌شود
        $msg = sendMessage($this->chat_id, $text, $keyboard, $this->message_id);
        
        if ($msg && isset($msg['result']['message_id'])) {
            $this->savePanelInfo($msg['result']['message_id'], 'tag');
        }
        
        return true;
    }
    
    /**
     * نمایش بیوگرافی تصادفی
     */
    private function showRandomBio() {
        if (!$this->hasAccess('bio')) return false;
        
        $bios = [
            "زندگی کوتاه است، لبخند بزن تا وقت داری!",
            "هر روز فرصتی تازه است.",
            "باش و بگذار دیگران باشند.",
            "رویاها را دنبال کن، موفقیت پیدا کن.",
            "زندگی یعنی ساختن خاطرات زیبا.",
            "هیچ‌وقت برای شروع دیر نیست.",
            "باور داشته باش، ایمان بیاور، بساز.",
            "امروز را بساز، فردا منتظر نباش.",
            "عشق، زندگی، خنده.",
            "قدرت در سکوت است."
        ];
        
        $random_bio = $bios[array_rand($bios)];
        
        $text = "✨ <b>بیوگرافی پیشنهادی:</b>\n\n";
        $text .= "💭 <code>{$random_bio}</code>\n\n";
        $text .= "📋 برای کپی روی متن کلیک کنید.";
        
        $this->reply($text);
        
        return true;
    }
    
    /**
     * نمایش بیوگرافی با کلیدواژه
     */
    private function showBioWithKeyword() {
        if (!$this->hasAccess('bio')) return false;
        
        preg_match('/^بیو (.+)$/u', $this->text, $matches);
        $keyword = $matches[1] ?? '';
        
        $bios = [
            "{$keyword} | زندگی زیباست",
            "♥️ {$keyword} ♥️",
            "✨ {$keyword} ✨",
            "{$keyword} برای همیشه",
            "🌟 {$keyword} 🌟",
            "عاشق {$keyword}",
            "{$keyword} | Be yourself",
            "🎯 {$keyword} 🎯"
        ];
        
        $text = "✨ <b>بیوگرافی‌های پیشنهادی با کلمه «{$keyword}»:</b>\n\n";
        
        foreach ($bios as $i => $bio) {
            $text .= ($i + 1) . ". <code>{$bio}</code>\n";
        }
        
        $text .= "\n📋 برای کپی روی هر متن کلیک کنید.";
        
        $this->reply($text);
        
        return true;
    }
    
    /**
     * نمایش فونت‌های مختلف
     */
    private function showFonts() {
        if (!$this->hasAccess('font')) return false;
        
        preg_match('/^فونت (.+)$/u', $this->text, $matches);
        $input = $matches[1] ?? '';
        
        if (empty($input)) {
            $this->reply("❌ لطفاً متن مورد نظر را وارد کنید.\nمثال: فونت سلام");
            return true;
        }
        
        // تشخیص فارسی یا انگلیسی
        $is_persian = preg_match('/[\x{0600}-\x{06FF}]/u', $input);
        
        if ($is_persian) {
            $fonts = $this->getPersianFonts($input);
            $title = "فارسی";
        } else {
            $fonts = $this->getEnglishFonts($input);
            $title = "انگلیسی";
        }
        
        $text = "🎨 <b>• لیست فونت {$title} {$input}</b>\n\n";
        
        foreach ($fonts as $i => $font) {
            $text .= ($i + 1) . "- <code>{$font}</code>\n";
        }
        
        $text .= "\n💡 برای کپی روی هر استایل کلیک کنید.";
        
        $this->reply($text);
        
        return true;
    }
    
    /**
     * نمایش لینک گروه
     */
    private function showGroupLink() {
        if (!$this->hasAccess('link')) return false;
        
        // دریافت تعداد اعضا
        $members_count = sendRequest('getChatMemberCount', ['chat_id' => $this->chat_id]);
        $total = $members_count['result'] ?? 0;
        
        $group_name = $this->message['chat']['title'];
        
        // دریافت لینک گروه
        $invite_link = sendRequest('exportChatInviteLink', ['chat_id' => $this->chat_id]);
        $link = $invite_link['result'] ?? null;
        
        if (!$link) {
            $this->reply("❌ نمی‌توان لینک گروه را دریافت کرد. ربات باید ادمین باشد.");
            return true;
        }
        
        $text = "🔗 <b>لینک گروه</b>\n\n";
        $text .= "👥 تعداد اعضا: {$total}\n";
        $text .= "📝 نام گروه: <a href='{$link}'>{$group_name}</a>";
        
        // دکمه اشتراک‌گذاری با لینک سالم گروه
        $share_text = urlencode("بیایید به گروه {$group_name} بپیوندید!");
        $keyboard = [
            'inline_keyboard' => [
                [
                    ['text' => '🔗 اشتراک‌گذاری لینک', 'url' => "https://t.me/share/url?url=" . urlencode($link) . "&text=" . $share_text]
                ]
            ]
        ];
        
        sendMessage($this->chat_id, $text, $keyboard, $this->message_id);
        
        return true;
    }
    
    /**
     * نمایش ساعت
     */
    private function showTime() {
        if (!$this->hasAccess('time')) return false;
        
        $jalali_date = jdate('l، j F Y');
        $jalali_time = jdate('H:i:s');
        
        $text = "🕐 <b>ساعت و تاریخ</b>\n\n";
        $text .= "📅 تاریخ: {$jalali_date}\n";
        $text .= "⏰ ساعت: {$jalali_time}";
        
        $this->reply($text);
        
        return true;
    }
    
    /**
     * سنجاق پیام
     */
    private function pinMessage() {
        if (!$this->hasAccess('pin')) return false;
        
        if (!isset($this->message['reply_to_message'])) {
            $this->reply("❌ لطفاً روی پیام مورد نظر ریپلای کنید.");
            return true;
        }
        
        $target_msg_id = $this->message['reply_to_message']['message_id'];
        
        $result = sendRequest('pinChatMessage', [
            'chat_id' => $this->chat_id,
            'message_id' => $target_msg_id,
            'disable_notification' => false
        ]);
        
        if ($result && $result['ok']) {
            $this->reply("📌 پیام با موفقیت سنجاق شد.");
        } else {
            $this->reply("❌ خطا در سنجاق کردن پیام. ربات باید ادمین باشد.");
        }
        
        return true;
    }
    
    /**
     * حذف سنجاق پیام
     */
    private function unpinMessage() {
        if (!$this->hasAccess('pin')) return false;
        
        if (isset($this->message['reply_to_message'])) {
            $target_msg_id = $this->message['reply_to_message']['message_id'];
            $result = sendRequest('unpinChatMessage', [
                'chat_id' => $this->chat_id,
                'message_id' => $target_msg_id
            ]);
        } else {
            $result = sendRequest('unpinAllChatMessages', [
                'chat_id' => $this->chat_id
            ]);
        }
        
        if ($result && $result['ok']) {
            $this->reply("✅ سنجاق با موفقیت حذف شد.");
        } else {
            $this->reply("❌ خطا در حذف سنجاق. ربات باید ادمین باشد.");
        }
        
        return true;
    }
    
    /**
     * تنظیم عکس گروه
     */
    private function setGroupPhoto() {
        if (!$this->hasAccess('photo')) return false;
        
        if (!isset($this->message['reply_to_message']['photo'])) {
            $this->reply("❌ لطفاً روی عکس مورد نظر ریپلای کنید.");
            return true;
        }
        
        $photo = end($this->message['reply_to_message']['photo']);
        $file_id = $photo['file_id'];
        
        // دریافت فایل
        $file_info = sendRequest('getFile', ['file_id' => $file_id]);
        
        if (!$file_info || !isset($file_info['result']['file_path'])) {
            $this->reply("❌ خطا در دریافت اطلاعات عکس.");
            return true;
        }
        
        $file_path = $file_info['result']['file_path'];
        $file_url = "https://api.telegram.org/file/bot" . BOT_TOKEN . "/{$file_path}";
        
        // دانلود عکس
        $photo_content = file_get_contents($file_url);
        $temp_file = sys_get_temp_dir() . '/group_photo_' . time() . '.jpg';
        file_put_contents($temp_file, $photo_content);
        
        // تنظیم عکس
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, "https://api.telegram.org/bot" . BOT_TOKEN . "/setChatPhoto");
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, [
            'chat_id' => $this->chat_id,
            'photo' => new CURLFile($temp_file)
        ]);
        $result = curl_exec($ch);
        curl_close($ch);
        
        unlink($temp_file);
        
        $result = json_decode($result, true);
        
        if ($result && $result['ok']) {
            $this->reply("✅ عکس گروه با موفقیت تغییر کرد.");
        } else {
            $this->reply("❌ خطا در تنظیم عکس. ربات باید ادمین باشد.");
        }
        
        return true;
    }
    
    /**
     * حذف عکس گروه
     */
    private function deleteGroupPhoto() {
        if (!$this->hasAccess('photo')) return false;
        
        $result = sendRequest('deleteChatPhoto', [
            'chat_id' => $this->chat_id
        ]);
        
        if ($result && $result['ok']) {
            $this->reply("✅ عکس گروه با موفقیت حذف شد.");
        } else {
            $this->reply("❌ خطا در حذف عکس. ربات باید ادمین باشد.");
        }
        
        return true;
    }
    
    /**
     * حذف یک پیام
     */
    private function deleteMessage() {
        if (!$this->hasAccess('delete')) return false;
        
        if (!isset($this->message['reply_to_message'])) {
            $this->reply("❌ لطفاً روی پیام مورد نظر ریپلای کنید.");
            return true;
        }
        
        $target_msg_id = $this->message['reply_to_message']['message_id'];
        
        // حذف پیام ریپلای شده
        deleteMessage($this->chat_id, $target_msg_id);
        
        // حذف دستور
        deleteMessage($this->chat_id, $this->message_id);
        
        return true;
    }
    
    /**
     * حذف چند پیام
     */
    private function deleteMultipleMessages() {
        if (!$this->hasAccess('delete')) return false;
        
        preg_match('/^حذف (\d+)$/u', $this->text, $matches);
        $count = intval($matches[1]);
        
        if ($count < 1 || $count > 100) {
            $this->reply("❌ تعداد باید بین 1 تا 100 باشد!");
            return true;
        }
        
        $deleted = 0;
        for ($i = 0; $i < $count; $i++) {
            $msg_id = $this->message_id - $i - 1;
            if ($msg_id > 0) {
                if (deleteMessage($this->chat_id, $msg_id)) {
                    $deleted++;
                }
            }
        }
        
        // حذف دستور
        deleteMessage($this->chat_id, $this->message_id);
        
        return true;
    }
    
    /**
     * حذف پیام‌های قدیمی
     */
    private function deleteMessagesByDate() {
        if (!$this->hasAccess('delete')) return false;
        
        preg_match('/^حذف (\d+) روز$/u', $this->text, $matches);
        $days = intval($matches[1]);
        
        if ($days < 1 || $days > 30) {
            $this->reply("❌ تعداد روز باید بین 1 تا 30 باشد!");
            return true;
        }
        
        $timestamp = time() - ($days * 24 * 60 * 60);
        
        $text = "⚠️ حذف پیام‌های {$days} روز گذشته ممکن است زمان‌بر باشد.\n";
        $text .= "این عملیات در پس‌زمینه انجام می‌شود.";
        
        $this->reply($text);
        
        // حذف دستور
        deleteMessage($this->chat_id, $this->message_id);
        
        return true;
    }
    
    // توابع کمکی
    
    private function isGroupActive() {
        $stmt = $this->db->prepare("SELECT is_active, expire_date FROM groups_data WHERE group_id = ?");
        $stmt->execute([$this->chat_id]);
        $result = $stmt->fetch();
        
        if (!$result || $result['is_active'] != 1) {
            return false;
        }
        
        // بررسی انقضا
        if ($result['expire_date'] && strtotime($result['expire_date']) < time()) {
            return false;
        }
        
        return true;
    }
    
    private function hasAccess($feature) {
        $rank = $this->getUserRankLevel($this->from_id);
        
        // ✅ اینجا اصلاح شد: دسترسی id از 0 (همه) به 1 (ویژه و بالاتر) تغییر کرد
        $access_map = [
            'id' => 1, // ویژه به بالا (سازنده، مالک، معاون، مشاور، ویژه)
            'nickname' => 1, 
            'origin' => 1,
            'tag' => 3, // مدیر به بالا
            'bio' => 1,
            'font' => 1,
            'link' => 0, // همه می‌توانند لینک بگیرند (اگر ربات ادمین باشد)
            'time' => 1,
            'pin' => 3,
            'photo' => 3,
            'delete' => 3
        ];
        
        $required_level = $access_map[$feature] ?? 999;
        
        return $rank >= $required_level;
    }
    
    private function getUserRankLevel($user_id) {
        if ($user_id == SUDO_ID) return 999;
        if ($this->isCreator($user_id)) return 6;
        if ($this->isOwner($user_id)) return 5;
        if ($this->isDeputy($user_id)) return 4;
        if ($this->isAdvisor($user_id)) return 3;
        if ($this->isVip($user_id)) return 1;
        return 0; // کاربر عادی و مدیر معمولی (اگر ویژه نباشد)
    }
    
    private function isCreator($user_id) {
        $stmt = $this->db->prepare("SELECT * FROM group_admins WHERE group_id = ? AND user_id = ? AND is_creator = 1");
        $stmt->execute([$this->chat_id, $user_id]);
        return $stmt->fetch() !== false;
    }
    
    private function isOwner($user_id) {
        $stmt = $this->db->prepare("SELECT * FROM mir_owners WHERE group_id = ? AND user_id = ?");
        $stmt->execute([$this->chat_id, $user_id]);
        return $stmt->fetch() !== false;
    }
    
    private function isDeputy($user_id) {
        $stmt = $this->db->prepare("SELECT * FROM mir_deputies WHERE group_id = ? AND user_id = ?");
        $stmt->execute([$this->chat_id, $user_id]);
        return $stmt->fetch() !== false;
    }
    
    private function isAdvisor($user_id) {
        $stmt = $this->db->prepare("SELECT * FROM mir_advisors WHERE group_id = ? AND user_id = ?");
        $stmt->execute([$this->chat_id, $user_id]);
        return $stmt->fetch() !== false;
    }
    
    private function isVip($user_id) {
        $stmt = $this->db->prepare("SELECT * FROM mir_vips WHERE group_id = ? AND user_id = ?");
        $stmt->execute([$this->chat_id, $user_id]);
        return $stmt->fetch() !== false;
    }
    
    private function getUserRank($user_id) {
        if ($user_id == SUDO_ID) return "👨‍💻 سودو";
        if ($this->isCreator($user_id)) return "👑 سازنده";
        if ($this->isOwner($user_id)) return "💎 وزیر اعظم";
        if ($this->isDeputy($user_id)) return "🎖 معاون";
        if ($this->isAdvisor($user_id)) return "📋 مشاور";
        if ($this->isVip($user_id)) return "⭐️ ویژه";
        return "👤 کاربر عادی";
    }
    
    private function getTargetUser() {
        if (isset($this->message['reply_to_message'])) {
            return $this->message['reply_to_message']['from'];
        }
        return $this->message['from'];
    }
    
    private function getMention($user) {
        $name = $user['first_name'] ?? 'کاربر';
        $user_id = $user['id'];
        return "<a href='tg://user?id={$user_id}'>{$name}</a>";
    }
    
    private function getUserData($user_id) {
        $stmt = $this->db->prepare("SELECT * FROM me_users WHERE group_id = ? AND user_id = ?");
        $stmt->execute([$this->chat_id, $user_id]);
        return $stmt->fetch();
    }
    
    private function updateUserField($user_id, $field, $value) {
        $stmt = $this->db->prepare("
            INSERT INTO me_users (group_id, user_id, {$field}) 
            VALUES (?, ?, ?) 
            ON DUPLICATE KEY UPDATE {$field} = ?
        ");
        $stmt->execute([$this->chat_id, $user_id, $value, $value]);
    }
    
    private function updateMessageStats($user) {
        $today = date('Y-m-d');
        
        $stmt = $this->db->prepare("
            INSERT INTO me_users (group_id, user_id, first_name, username, daily_messages, total_messages, last_message_date) 
            VALUES (?, ?, ?, ?, 1, 1, ?) 
            ON DUPLICATE KEY UPDATE 
                first_name = ?,
                username = ?,
                daily_messages = IF(last_message_date = ?, daily_messages + 1, 1),
                total_messages = total_messages + 1,
                last_message_date = ?,
                last_seen = NOW()
        ");
        
        $stmt->execute([
            $this->chat_id,
            $user['id'],
            $user['first_name'] ?? 'کاربر',
            $user['username'] ?? null,
            $today,
            $user['first_name'] ?? 'کاربر',
            $user['username'] ?? null,
            $today,
            $today
        ]);
    }
    
    // فونت‌ها
    private function getPersianFonts($text) {
        return [
            "مَِــَِ{$text}َِ",
            "ۘۘ{$text}ـ ۘۘ",
            "مُِـٖٖـۘۘـُِ{$text}ُِـٖٖـۘۘـُِدَّ",
            "{$text}ـــــدّ",
            "{$text}ٖٖــٍ‌ـۘۘــ{$text}ٖٖــٍ‌ـۘۘــدۘۘـ",
            "{$text}ُِـٖٖـۘۘـُِ{$text}ُِـٖٖـۘۘـُِدَّ",
            "{$text}ـٰٰـٰٰ{$text}ـٰٰـٰٰــدٍٕ",
            "{$text}ـؒؔ◌‌‌ࢪ{$text}ـؒؔ◌‌‌ࢪـ‌َد",
            "{$text}ـٜ٘ـٍٜـٜۘـٜۘـٍٍـٜٜ{$text}ـٜ٘ـٍٜـٜۘـٜۘـٍٍـٜٜ  ًًد''",
            "{$text}ـ﹏ـ{$text}ـ﹏ـد",
            "{$text}ـ۪ٜـ۪ٜـ۪ٜـ۪ٜـ۪ٜـ۪ٜ‌{$text}ـ۪ٜـ۪ٜـ۪ٜـ۪ٜـ۪ٜـ۪ٜ‌د۪ٜ",
            "{$text}ــؒؔ✫ؒؔـ ҉๏‌‌๏ًٍ{$text}ــؒؔ✫ؒؔـ ҉๏‌‌๏ًٍَؔد",
            "{$text}ـٖٖـۘۘ℘ـʘ‌‌{$text}ـٖٖـۘۘ℘ـʘ‌‌د",
            "{$text}ـ෴ِْ{$text}ـ෴ِْد",
            "{$text}ـًٍʘًٍʘـ{$text}ـًٍʘًٍʘـدََ",
            "{$text}ـؒؔـٓٓـؒؔ◌‌‌◌{$text}ـؒؔـٓٓـؒؔ◌‌‌◌د",
            "{$text}ـؒؔـٜٜঊٌٍـ↯ـٜٜـٍٍـ{$text}ـؒؔـٜٜঊٌٍـ↯ـٜٜـٍٍـدٜٜঊٌٍ",
            "{$text}ـؒؔـؒؔـَؔ௸{$text}ـؒؔـؒؔـَؔ௸د",
            "{$text}ــؕؕـٜٜـٜٜ✿{$text}ــؕؕـٜٜـٜٜ✿دٜٜ",
            "{$text}َِـَِـَِـَِـَِـَِـَِـَِـَِـَِـَِـَِـَِـَ{$text}َِـَِـَِـَِـَِـَِـَِـَِـَِـَِـَِـَِـَِـَدَِ",
            "{$text}ــؕؕـٜٜـٜٜ✿ٜٜـٜٜـٜٜـ{$text}ــؕؕـٜٜـٜٜ✿ٜٜـٜٜـٜٜـدٜٜ",
            "{$text}ـٰٖـۘۘـــٍٰـ{$text}ـٰٖـۘۘـــٍٰـدٰٖ",
            "{$text}ـ‌‌ـ۪ٜـ۪ٜ❀{$text}ـ‌‌ـ۪ٜـ۪ٜ❀۪ٜ❀د",
            "{$text}ـ℘ू{$text}ـ℘ूٜ۪◌د‌‌✾",
            "{$text}ـैـ۪ٜـ۪ٜـ۪ٜ❀‌‌ــؒؔ{$text}ـैـ۪ٜـ۪ٜـ۪ٜ❀‌‌ــؒؔ❀'‌‌دَّ'",
            "{$text}ـ‌‌ــؒؔـ‌‌ـ{$text}ـ‌‌ــؒؔـ‌‌ـ❁'د۠۠",
            "{$text}ٰٖـٰٖ℘ـ‌َ✾ـ{$text}ٰٖـٰٖ℘ـ‌َ✾ـدٰٖـٰٖ℘ـ‌َ✾ـ",
            "{$text}❈ۣۣـ🍁ـ{$text}❈ۣۣـ🍁ـد❈ۣۣـ🍁ـ",
            "{$text}ْஓ‌ঠৡ{$text}ْஓ‌ঠৡدْஓ‌ঠৡ",
            "{$text}ـೄ{$text}ـೄدٍٖ❦",
            "✮{$text}ًٍـًٍـَؔ✯ًٍ✮{$text}ًٍـًٍـَؔ✯ًٍ✮ًٍد",
            "{$text}ـؒؔـؒؔـ۪۪ـؒؔـؒؔـ‌ົ້◌ฺฺಹ۪۪{$text}ـؒؔـؒؔـ۪۪ـؒؔـؒؔـ‌ົ້◌ฺฺಹ۪۪✯د✯",
            "{$text}ٰٓـؒؔـ۪۪ঊ۝{$text}ٰٓـؒؔـ۪۪ঊ۝۪۪دؓ",
            "{$text}۪۟ـ۟۟✶ًٍـ۟ـًٍـ۪۟ـ۟ـًٍــ۪۟ـ۟۟ـً۟ــٗؔـٗؔ✰‌‌℘ِِ{$text}۪۟ـ۟۟✶ًٍـ۟ـًٍـ۪۟ـ۟ـًٍــ۪۟ـ۟۟ـً۟ــٗؔـٗؔ✰‌‌℘ِِ✰‌‌د℘ِِ",
            "{$text}ؔؑــَؔـَؔ ـَؔ สฺฺŗـَؔ๛ٖؔ{$text}ؔؑــَؔـَؔ ـَؔ สฺฺŗـَؔ๛ٖؔ❂ؔؑـَؔد۪๛ٖؔ",
            "{$text}ــ ོꯨ҉ــؒؔ҉:ــــ{$text}ــ ོꯨ҉ــؒؔ҉:ــــدــ ོꯨ҉ــؒؔ҉:ــــ",
            "{$text}ؔؑـَؔ ـؔؑـَؔ๛ٖؔ{$text}ؔؑـَؔ ـؔؑـَؔ๛ٖؔد",
            "{$text}ـ‌‌ـ‌‌ـ‌‌ـ✺۠۠➤{$text}ـ‌‌ـ‌‌ـ‌‌ـ✺۠۠➤د✺۠۠➤",
            "{$text}ٖـٖٗ⸭ـٖٗـٖٗ⸭{$text}ٖـٖٗ⸭ـٖٗـٖٗ⸭ٖٗ⸭ـٖٖٗـٖٗد⸭",
            "{$text}ـٍ‌ــٍ‌ــٍ‌❉{$text}ـٍ‌ــٍ‌ــٍ‌❉دٍ‌❉",
            "{$text}ـْْـْْـْْ/ْْ{$text}ـْْـْْـْْ/ْْـْْـْْـدْْ/",
            "↜{$text}ٍٍـُِ➲ِِனُِ↜{$text}ٍٍـُِ➲ِِனُِ↜ٍٍـُِد➲ِِனُِ",
            "{$text}ـ‌‌ــ‌‌◕‌‌₰‌‌◚‌‌₰‌‌{$text}ـ‌‌ــ‌‌◕‌‌₰‌‌◚‌‌₰‌‌د‌‌◕‌‌₰",
            "{$text}ـٰٓـًً◑ِّ◑ًً{$text}ـٰٓـًً◑ِّ◑ًًد◑ِّ◑ًً",
            "{$text}ـ‌‌ـ‌‌ـ‌‌✭{$text}ـ‌‌ـ‌‌ـ‌‌✭‌‌د✭",
            "{$text}ـًٍـؒؔـؒؔ⸙ؒৡ✪{$text}ـًٍـؒؔـؒؔ⸙ؒৡ✪د✪",
            "{$text}ـ◎۪۪❖ु{$text}ـ◎۪۪❖ुद◎۪۪❖ु",
            "{$text}ـٌْ‌ٍٖـ۪ٓ{$text}ـٌْ‌ٍٖـ۪ٓदुـ۪ٓ",
            "ِْ{$text}ـِْ❉ِْ{$text}ـِْ❉ِْدـِْ❉",
            "[ِْـ{$text}ِْـِْ❉ِْـِْ[ِْـ{$text}ِْـِْ❉ِْـِْ[ِْـِْدـِْ❉ِْـِْ",
            "❅{$text}ـ۪۪ـؒؔـؒؔـ۪۪ـؒؔـؒؔ❢❅{$text}ـ۪۪ـؒؔـؒؔـ۪۪ـؒؔـؒؔ❢❅ـ۪۪ـؒؔـدؒؔـ۪۪ـؒؔـؒؔ❢",
            "{$text}ٖؒـؒؔـٰٰـٖٖ{$text}ٖؒـؒؔـٰٰـٖٖدٖؒـؒؔـٰٰـٖٖ"
        ];
    }
    
    private function getEnglishFonts($text) {
        $lower = strtolower($text);
        $upper = strtoupper($text);
        
        $fonts = [];
        
        $fonts[] = $this->toCircled($lower);
        $fonts[] = $this->toParenthesized($upper);
        $fonts[] = "мм∂";
        $fonts[] = $this->toBold($upper);
        $fonts[] = $this->toBoldItalic($upper);
        $fonts[] = $this->toItalic($upper);
        $fonts[] = $this->toSansBold($upper);
        $fonts[] = $this->toSans($upper);
        $fonts[] = $this->toDoubleStruck($upper);
        $fonts[] = $this->toFraktur($upper);
        $fonts[] = $this->toMonospace($upper);
        $fonts[] = $this->toSansItalic($upper);
        $fonts[] = str_replace(str_split($lower), ['M‌‌', 'M‌‌', 'D‌‌'], $lower);
        $fonts[] = str_replace(str_split($lower), ['M‌‌', 'M‌‌', 'D‌‌'], $lower);
        $fonts[] = str_replace(str_split($lower), ['M‌‌', 'M‌‌', 'D‌‌'], $lower);
        $fonts[] = str_replace(str_split($lower), ['M‌‌', 'M‌‌', 'D‌‌'], $lower);
        $fonts[] = str_replace(str_split($lower), ['M‌‌', 'M‌‌', 'D‌‌'], $lower);
        $fonts[] = str_replace(str_split($lower), ['M‌‌', 'M‌‌', 'D‌‌'], $lower);
        $fonts[] = "从从刀";
        $fonts[] = "ጠጠሏ";
        $fonts[] = "ᎷᎷᎠ";
        $fonts[] = "ᛖᛖΔ";
        $fonts[] = "ꂵꂵꅓ";
        $fonts[] = "ᗰᗰᗪ";
        $fonts[] = "ᗰᗰᗞ";
        $fonts[] = "ℳℳⅅ";
        $fonts[] = "ᚥᚥᚦ";
        $fonts[] = $this->toScript($upper);
        $fonts[] = "ℳℳ𝒟";
        $fonts[] = $this->toBlackboard($upper);
        $fonts[] = "ＭＭＤ";
        $fonts[] = "ᴍᴍᴅ";
        $fonts[] = "ᴹᴹᴰ";
        $fonts[] = "🅼🅼🅳";
        $fonts[] = "Ⓜ️Ⓜ️Ⓓ";
        $fonts[] = "🄼🄼🄳";
        $fonts[] = "🅜🅜🅓";
        $fonts[] = "🇲 🇲 🇩";
        $fonts[] = "ммd";
        $fonts[] = "ммđ";
        $fonts[] = "ʍʍժ";
        $fonts[] = "ɱɱძ";
        $fonts[] = $lower;
        $fonts[] = "๓๓๔";
        $fonts[] = "ᄊᄊÐ";
        $fonts[] = "ʍʍδ";
        $fonts[] = "ოოժ";
        $fonts[] = "ɱɱƊ";
        $fonts[] = "mmÐ";
        $fonts[] = "ммb";
        $fonts[] = "‌m‌‌m‌‌d‌";
        $fonts[] = "мм∂";
        $fonts[] = "ммɗ";
        $fonts[] = "ммd";
        $fonts[] = "ɱɱԃ";
        $fonts[] = "ммđ";
        $fonts[] = "ɱɱđ";
        $fonts[] = "ʍʍժ";
        $fonts[] = "ოოძ";
        $fonts[] = "ɱɱძ";
        $fonts[] = $lower;
        $fonts[] = "ммÞ";
        $fonts[] = "ϻϻd";
        $fonts[] = "๓๓๔";
        $fonts[] = $upper;
        $fonts[] = "ᄊᄊÐ";
        $fonts[] = "ʍʍδ";
        $fonts[] = "๓๓໓";
        $fonts[] = "ոոժ";
        $fonts[] = "♏♏Ð";
        $fonts[] = "ɱɱƊ";
        $fonts[] = "mmÐ";
        $fonts[] = "ʍʍԺ";
        $fonts[] = "MMÐ";
        $fonts[] = "ммb";
        $fonts[] = $upper;
        $fonts[] = "‌m‌‌m‌‌d‌";
        $fonts[] = "‌m‌‌m‌‌d‌";
        $fonts[] = "‌m‌‌m‌‌d‌";
        $fonts[] = "‌m‌‌m‌‌d‌";
        $fonts[] = $upper;
        $fonts[] = $lower;
        $fonts[] = $this->toCircled($lower);
        $fonts[] = "ммⅾ";
        $fonts[] = "мм∂";
        $fonts[] = "ммḓ";
        $fonts[] = "☺️☺️℮";
        $fonts[] = "ღღḓ";
        $fonts[] = "ღღ∂";
        $fonts[] = "μμ∂";
        $fonts[] = "мм∂";
        $fonts[] = "ммḓ";
        $fonts[] = "ⓜⓜ∂";
        $fonts[] = "μμ∂";
        $fonts[] = "ⓜⓜⅾ";
        $fonts[] = "ммⅾ";
        $fonts[] = "μμⅾ";
        $fonts[] = "๓๓๔";
        $fonts[] = "օօբ";
        $fonts[] = "ɱɱძ";
        $fonts[] = "mmÐ";
        $fonts[] = "ɱɱɖ";
        $fonts[] = "ммÞ";
        $fonts[] = "ʍʍԺ";
        $fonts[] = "௱௱Ð";
        $fonts[] = "ɱɱD";
        $fonts[] = "ммb";
        $fonts[] = "MMÐ";
        $fonts[] = "mmð";
        $fonts[] = $this->toParenthesized($upper);
        $fonts[] = $upper;
        $fonts[] = "ммđ";
        $fonts[] = "ммɔ";
        $fonts[] = "【m】【m】【d】";
        $fonts[] = "‌m‌‌‌m‌‌‌‌d‌‌‌";
        $fonts[] = "‌m‌‌‌‌m‌‌‌‌‌d‌‌";
        $fonts[] = "‌‌‌‌m‌‌‌‌‌‌‌‌‌m‌‌‌‌‌‌‌d‌‌‌‌";
        $fonts[] = "ะmะmะd";
        $fonts[] = "‌m‌‌m‌‌d‌";
        $fonts[] = "‌‌m‌‌m‌‌d";
        $fonts[] = "‌‌m‌‌‌‌m‌‌‌d‌‌";
        $fonts[] = "‌‌m‌‌‌‌m‌‌‌‌d‌‌";
        $fonts[] = "‌m‌‌m‌d‌‌";
        $fonts[] = "‌m‌‌m‌‌d‌";
        $fonts[] = "♏♏Ð";
        
        return $fonts;
    }
    
    // توابع تبدیل Unicode
    private function toCircled($t) {
        $map = ['a'=>'ⓐ','b'=>'ⓑ','c'=>'ⓒ','d'=>'ⓓ','e'=>'ⓔ','f'=>'ⓕ','g'=>'ⓖ','h'=>'ⓗ','i'=>'ⓘ','j'=>'ⓙ','k'=>'ⓚ','l'=>'ⓛ','m'=>'ⓜ','n'=>'ⓝ','o'=>'ⓞ','p'=>'ⓟ','q'=>'ⓠ','r'=>'ⓡ','s'=>'ⓢ','t'=>'ⓣ','u'=>'ⓤ','v'=>'ⓥ','w'=>'ⓦ','x'=>'ⓧ','y'=>'ⓨ','z'=>'ⓩ'];
        return strtr($t, $map);
    }
    private function toParenthesized($t) {
        $map = ['A'=>'⒜','B'=>'⒝','C'=>'⒞','D'=>'⒟','E'=>'⒠','F'=>'⒡','G'=>'⒢','H'=>'⒣','I'=>'⒤','J'=>'⒥','K'=>'⒦','L'=>'⒧','M'=>'⒨','N'=>'⒩','O'=>'⒪','P'=>'⒫','Q'=>'⒬','R'=>'⒭','S'=>'⒮','T'=>'⒯','U'=>'⒰','V'=>'⒱','W'=>'⒲','X'=>'⒳','Y'=>'⒴','Z'=>'⒵'];
        return strtr($t, $map);
    }
    private function toBold($t) {
        $map = ['A'=>'𝐀','B'=>'𝐁','C'=>'𝐂','D'=>'𝐃','E'=>'𝐄','F'=>'𝐅','G'=>'𝐆','H'=>'𝐇','I'=>'𝐈','J'=>'𝐉','K'=>'𝐊','L'=>'𝐋','M'=>'𝐌','N'=>'𝐍','O'=>'𝐎','P'=>'𝐏','Q'=>'𝐐','R'=>'𝐑','S'=>'𝐒','T'=>'𝐓','U'=>'𝐔','V'=>'𝐕','W'=>'𝐖','X'=>'𝐗','Y'=>'𝐘','Z'=>'𝐙'];
        return strtr($t, $map);
    }
    private function toBoldItalic($t) {
        $map = ['A'=>'𝑨','B'=>'𝑩','C'=>'𝑪','D'=>'𝑫','E'=>'𝑬','F'=>'𝑭','G'=>'𝑮','H'=>'𝑯','I'=>'𝑰','J'=>'𝑱','K'=>'𝑲','L'=>'𝑳','M'=>'𝑴','N'=>'𝑵','O'=>'𝑶','P'=>'𝑷','Q'=>'𝑸','R'=>'𝑹','S'=>'𝑺','T'=>'𝑻','U'=>'𝑼','V'=>'𝑽','W'=>'𝑾','X'=>'𝑿','Y'=>'𝒀','Z'=>'𝒁'];
        return strtr($t, $map);
    }
    private function toItalic($t) {
        $map = ['A'=>'𝐴','B'=>'𝐵','C'=>'𝐶','D'=>'𝐷','E'=>'𝐸','F'=>'𝐹','G'=>'𝐺','H'=>'𝐻','I'=>'𝐼','J'=>'𝐽','K'=>'𝐾','L'=>'𝐿','M'=>'𝑀','N'=>'𝑁','O'=>'𝑂','P'=>'𝑃','Q'=>'𝑄','R'=>'𝑅','S'=>'𝑆','T'=>'𝑇','U'=>'𝑈','V'=>'𝑉','W'=>'𝑊','X'=>'𝑋','Y'=>'𝑌','Z'=>'𝑍'];
        return strtr($t, $map);
    }
    private function toSansBold($t) {
        $map = ['A'=>'𝗔','B'=>'𝗕','C'=>'𝗖','D'=>'𝗗','E'=>'𝗘','F'=>'𝗙','G'=>'𝗚','H'=>'𝗛','I'=>'𝗜','J'=>'𝗝','K'=>'𝗞','L'=>'𝗟','M'=>'𝗠','N'=>'𝗡','O'=>'𝗢','P'=>'𝗣','Q'=>'𝗤','R'=>'𝗥','S'=>'𝗦','T'=>'𝗧','U'=>'𝗨','V'=>'𝗩','W'=>'𝗪','X'=>'𝗫','Y'=>'𝗬','Z'=>'𝗭'];
        return strtr($t, $map);
    }
    private function toSans($t) {
        $map = ['A'=>'𝖠','B'=>'𝖡','C'=>'𝖢','D'=>'𝖣','E'=>'𝖤','F'=>'𝖥','G'=>'𝖦','H'=>'𝖧','I'=>'𝖨','J'=>'𝖩','K'=>'𝖪','L'=>'𝖫','M'=>'𝖬','N'=>'𝖭','O'=>'𝖮','P'=>'𝖯','Q'=>'𝖰','R'=>'𝖱','S'=>'𝖲','T'=>'𝖳','U'=>'𝖴','V'=>'𝖵','W'=>'𝖶','X'=>'𝖷','Y'=>'𝖸','Z'=>'𝖹'];
        return strtr($t, $map);
    }
    private function toDoubleStruck($t) {
        $map = ['A'=>'𝔸','B'=>'𝔹','C'=>'ℂ','D'=>'𝔻','E'=>'𝔼','F'=>'𝔽','G'=>'𝔾','H'=>'ℍ','I'=>'𝕀','J'=>'𝕁','K'=>'𝕂','L'=>'𝕃','M'=>'𝕄','N'=>'ℕ','O'=>'𝕆','P'=>'ℙ','Q'=>'ℚ','R'=>'ℝ','S'=>'𝕊','T'=>'𝕋','U'=>'𝕌','V'=>'𝕍','W'=>'𝕎','X'=>'𝕏','Y'=>'𝕐','Z'=>'ℤ'];
        return strtr($t, $map);
    }
    private function toFraktur($t) {
        $map = ['A'=>'𝔄','B'=>'𝔅','C'=>'ℭ','D'=>'𝔇','E'=>'𝔈','F'=>'𝔉','G'=>'𝔊','H'=>'ℌ','I'=>'ℑ','J'=>'𝔍','K'=>'𝔎','L'=>'𝔏','M'=>'𝔐','N'=>'𝔑','O'=>'𝔒','P'=>'𝔓','Q'=>'𝔔','R'=>'ℜ','S'=>'𝔖','T'=>'𝔗','U'=>'𝔘','V'=>'𝔙','W'=>'𝔚','X'=>'𝔛','Y'=>'𝔜','Z'=>'ℨ'];
        return strtr($t, $map);
    }
    private function toMonospace($t) {
        $map = ['A'=>'𝙰','B'=>'𝙱','C'=>'𝙲','D'=>'𝙳','E'=>'𝙴','F'=>'𝙵','G'=>'𝙶','H'=>'𝙷','I'=>'𝙸','J'=>'𝙹','K'=>'𝙺','L'=>'𝙻','M'=>'𝙼','N'=>'𝙽','O'=>'𝙾','P'=>'𝙿','Q'=>'𝚀','R'=>'𝚁','S'=>'𝚂','T'=>'𝚃','U'=>'𝚄','V'=>'𝚅','W'=>'𝚆','X'=>'𝚇','Y'=>'𝚈','Z'=>'𝚉'];
        return strtr($t, $map);
    }
    private function toSansItalic($t) {
        $map = ['A'=>'𝘈','B'=>'𝘉','C'=>'𝘊','D'=>'𝘋','E'=>'𝘌','F'=>'𝘍','G'=>'𝘎','H'=>'𝘏','I'=>'𝘐','J'=>'𝘑','K'=>'𝘒','L'=>'𝘓','M'=>'𝘔','N'=>'𝘕','O'=>'𝘖','P'=>'𝘗','Q'=>'𝘘','R'=>'𝘙','S'=>'𝘚','T'=>'𝘛','U'=>'𝘜','V'=>'𝘝','W'=>'𝘞','X'=>'𝘟','Y'=>'𝘠','Z'=>'𝘡'];
        return strtr($t, $map);
    }
    private function toScript($t) {
        $map = ['A'=>'𝒜','B'=>'ℬ','C'=>'𝒞','D'=>'𝒟','E'=>'ℰ','F'=>'ℱ','G'=>'𝒢','H'=>'ℋ','I'=>'ℐ','J'=>'𝒥','K'=>'𝒦','L'=>'ℒ','M'=>'𝓜','N'=>'𝒩','O'=>'𝒪','P'=>'𝒫','Q'=>'𝒬','R'=>'ℛ','S'=>'𝒮','T'=>'𝒯','U'=>'𝒰','V'=>'𝒱','W'=>'𝒲','X'=>'𝒳','Y'=>'𝒴','Z'=>'𝒵'];
        return strtr($t, $map);
    }
    private function toBlackboard($t) {
        $map = ['A'=>'𝔸','B'=>'𝔹','C'=>'ℂ','D'=>'𝔻','E'=>'𝔼','F'=>'𝔽','G'=>'𝔾','H'=>'ℍ','I'=>'𝕀','J'=>'𝕁','K'=>'𝕂','L'=>'𝕃','M'=>'𝕄','N'=>'ℕ','O'=>'𝕆','P'=>'ℙ','Q'=>'ℚ','R'=>'ℝ','S'=>'𝕊','T'=>'𝕋','U'=>'𝕌','V'=>'𝕍','W'=>'𝕎','X'=>'𝕏','Y'=>'𝕐','Z'=>'ℤ'];
        return strtr($t, $map);
    }
    
    private function savePanelInfo($message_id, $type) {
        $stmt = $this->db->prepare("INSERT INTO me_open_panels (group_id, message_id, panel_type) VALUES (?, ?, ?)");
        $stmt->execute([$this->chat_id, $message_id, $type]);
    }
    
    private function reply($text) {
        sendMessage($this->chat_id, $text, null, $this->message_id);
    }
}

/**
 * پردازش دستورات ME
 */
function processMeCommand($message) {
    $me = new MeSystem($message);
    return $me->process();
}

/**
 * مدیریت callback های ME
 */
function handleMeCallback($callback) {
    $data = $callback['data'];
    $message = $callback['message'];
    $chat_id = $message['chat']['id'];
    $message_id = $message['message_id'];
    
    // استخراج message_id از callback_data
    if (preg_match('/me_tag_(members|admins|active)_(\d+)/', $data, $matches)) {
        $type = $matches[1];
        $reply_to_msg_id = intval($matches[2]);
        
        if ($type == 'members') {
            tagMembers($chat_id, $reply_to_msg_id, $message_id, $callback['id']);
            return true;
        }
        if ($type == 'admins') {
            tagAdmins($chat_id, $reply_to_msg_id, $message_id, $callback['id']);
            return true;
        }
        if ($type == 'active') {
            tagActiveUsers($chat_id, $reply_to_msg_id, $message_id, $callback['id']);
            return true;
        }
    }
    
    if ($data == 'me_close_panel') {
        editMessage($chat_id, $message_id, "✅ <b>پنل در صورت استفاده نکردن بسته شد</b>");
        sendRequest('answerCallbackQuery', [
            'callback_query_id' => $callback['id'],
            'text' => 'پنل بسته شد'
        ]);
        return true;
    }
    
    return false;
}

/**
 * تگ کردن تمام اعضای گروه
 * اصلاح شده: ریپلای هوشمند + حذف پیام پایان + فرمت 5 تایی با نقطه
 */
function tagMembers($chat_id, $reply_to_msg_id, $panel_msg_id, $callback_id) {
    sendRequest('answerCallbackQuery', [
        'callback_query_id' => $callback_id,
        'text' => '⏳ در حال تگ کردن...'
    ]);
    
    // حذف پنل
    deleteMessage($chat_id, $panel_msg_id);
    
    // دریافت تمام اعضا از دیتابیس (تا 2000 نفر)
    $db = Database::instance()->pdo();
    $stmt = $db->prepare("SELECT DISTINCT user_id, first_name FROM me_users WHERE group_id = ? LIMIT 2000");
    $stmt->execute([$chat_id]);
    $users = $stmt->fetchAll();
    
    if (empty($users)) {
        sendMessage($chat_id, "❌ هیچ کاربری یافت نشد!", null, $reply_to_msg_id);
        return;
    }
    
    // ساخت منشن‌ها با نام واقعی
    $mentions = [];
    foreach ($users as $user) {
        $name = htmlspecialchars($user['first_name']) ?: 'کاربر';
        $mentions[] = "<a href='tg://user?id={$user['user_id']}'>{$name}</a>";
    }
    
    // تقسیم به گروه‌های 5 تایی و ارسال
    $chunks = array_chunk($mentions, 5);
    foreach ($chunks as $chunk) {
        $text = implode(' • ', $chunk);
        // ارسال ریپلای روی پیام هدف
        sendMessage($chat_id, $text, null, $reply_to_msg_id);
        // وقفه برای جلوگیری از محدودیت
        usleep(700000); 
    }
    
    // پیام پایان حذف شد
}

/**
 * تگ کردن ادمین‌ها و سازنده
 * اصلاح شده: ریپلای هوشمند + حذف پیام پایان + فرمت 5 تایی با نقطه
 */
function tagAdmins($chat_id, $reply_to_msg_id, $panel_msg_id, $callback_id) {
    sendRequest('answerCallbackQuery', [
        'callback_query_id' => $callback_id,
        'text' => '⏳ در حال تگ کردن...'
    ]);
    
    // حذف پنل
    deleteMessage($chat_id, $panel_msg_id);
    
    // دریافت ادمین‌ها از تلگرام
    $admins = sendRequest('getChatAdministrators', ['chat_id' => $chat_id]);
    
    if (!$admins || !isset($admins['result'])) {
        sendMessage($chat_id, "❌ خطا در دریافت لیست ادمین‌ها!", null, $reply_to_msg_id);
        return;
    }
    
    $mentions = [];
    foreach ($admins['result'] as $admin) {
        $user = $admin['user'];
        if (!$user['is_bot']) {
            $name = htmlspecialchars($user['first_name']) ?: 'کاربر';
            $mentions[] = "<a href='tg://user?id={$user['id']}'>{$name}</a>";
        }
    }
    
    if (empty($mentions)) {
        sendMessage($chat_id, "❌ هیچ ادمینی یافت نشد!", null, $reply_to_msg_id);
        return;
    }
    
    // ارسال تگ‌ها
    $chunks = array_chunk($mentions, 5);
    foreach ($chunks as $chunk) {
        $text = implode(' • ', $chunk);
        sendMessage($chat_id, $text, null, $reply_to_msg_id);
        usleep(700000);
    }
    
    // پیام پایان حذف شد
}

/**
 * تگ کردن کاربران فعال (تا 300 نفر)
 * اصلاح شده: ریپلای هوشمند + حذف پیام پایان + فرمت 5 تایی با نقطه
 */
function tagActiveUsers($chat_id, $reply_to_msg_id, $panel_msg_id, $callback_id) {
    sendRequest('answerCallbackQuery', [
        'callback_query_id' => $callback_id,
        'text' => '⏳ در حال تگ کردن...'
    ]);
    
    // حذف پنل
    deleteMessage($chat_id, $panel_msg_id);
    
    $db = Database::instance()->pdo();
    
    // کاربران فعال: کسانی که پیام داده‌اند (تا 300 نفر)
    $stmt = $db->prepare("SELECT user_id, first_name FROM me_users WHERE group_id = ? AND total_messages > 0 ORDER BY last_seen DESC LIMIT 300");
    $stmt->execute([$chat_id]);
    $users = $stmt->fetchAll();
    
    if (empty($users)) {
        sendMessage($chat_id, "❌ هیچ کاربر فعالی یافت نشد!", null, $reply_to_msg_id);
        return;
    }
    
    $mentions = [];
    foreach ($users as $user) {
        $name = htmlspecialchars($user['first_name']) ?: 'کاربر';
        $mentions[] = "<a href='tg://user?id={$user['user_id']}'>{$name}</a>";
    }
    
    // ارسال تگ‌ها
    $chunks = array_chunk($mentions, 5);
    foreach ($chunks as $chunk) {
        $text = implode(' • ', $chunk);
        sendMessage($chat_id, $text, null, $reply_to_msg_id);
        usleep(700000);
    }
    
    // پیام پایان حذف شد
}

/**
 * حذف اطلاعات کاربر (وقتی ترک می‌کند)
 */
function deleteUserData($chat_id, $user_id) {
    $db = Database::instance()->pdo();
    
    try {
        // حذف اطلاعات کاربر
        $stmt = $db->prepare("DELETE FROM me_users WHERE group_id = ? AND user_id = ?");
        $stmt->execute([$chat_id, $user_id]);
        
        return true;
    } catch(PDOException $e) {
        error_log("Delete User Data Error: " . $e->getMessage());
        return false;
    }
}

/**
 * بستن پنل‌های باز (Cron Job)
 */
function closeExpiredPanels() {
    $db = Database::instance()->pdo();
    
    // پنل‌هایی که بیش از 5 دقیقه باز هستند
    $stmt = $db->prepare("
        SELECT * FROM me_open_panels 
        WHERE opened_at < DATE_SUB(NOW(), INTERVAL 5 MINUTE)
    ");
    $stmt->execute();
    $panels = $stmt->fetchAll();
    
    $closed_count = 0;
    foreach ($panels as $panel) {
        // اطمینان از وجود توابع
        if (function_exists('editMessage')) {
            $result = editMessage(
                $panel['group_id'], 
                $panel['message_id'], 
                "✅ <b>پنل در صورت استفاده نکردن بسته شد</b>"
            );
            
            if ($result) {
                $stmt = $db->prepare("DELETE FROM me_open_panels WHERE id = ?");
                $stmt->execute([$panel['id']]);
                $closed_count++;
            }
        }
    }
    
    return $closed_count;
}

/**
 * ریست آمار روزانه (Cron Job - هر شب 00:00)
 */
function resetDailyStats() {
    $db = Database::instance()->pdo();
    
    $yesterday = date('Y-m-d', strtotime('-1 day'));
    
    $stmt = $db->prepare("
        UPDATE me_users 
        SET daily_messages = 0 
        WHERE last_message_date < ?
    ");
    $stmt->execute([$yesterday]);
    
    return $stmt->rowCount();
}

/**
 * ایجاد جداول MIR اگر وجود ندارند
 * در صورت تداخل با فایل اصلی، نادیده گرفته می‌شود
 */
if (!function_exists('createMirTables')) {
    function createMirTables() {
        $db = Database::instance()->pdo();
        
        // جدول مالکین (وزیر اعظم)
        $db->exec("CREATE TABLE IF NOT EXISTS mir_owners (
            id INT AUTO_INCREMENT PRIMARY KEY,
            group_id BIGINT NOT NULL,
            user_id BIGINT NOT NULL,
            user_name VARCHAR(255),
            username VARCHAR(255),
            added_by BIGINT NOT NULL,
            added_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            UNIQUE KEY unique_owner (group_id, user_id),
            INDEX idx_group_user (group_id, user_id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci");
        
        // جدول معاونین
        $db->exec("CREATE TABLE IF NOT EXISTS mir_deputies (
            id INT AUTO_INCREMENT PRIMARY KEY,
            group_id BIGINT NOT NULL,
            user_id BIGINT NOT NULL,
            user_name VARCHAR(255),
            username VARCHAR(255),
            added_by BIGINT NOT NULL,
            added_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            UNIQUE KEY unique_deputy (group_id, user_id),
            INDEX idx_group_user (group_id, user_id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci");
        
        // جدول مشاورین
        $db->exec("CREATE TABLE IF NOT EXISTS mir_advisors (
            id INT AUTO_INCREMENT PRIMARY KEY,
            group_id BIGINT NOT NULL,
            user_id BIGINT NOT NULL,
            user_name VARCHAR(255),
            username VARCHAR(255),
            added_by BIGINT NOT NULL,
            added_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            UNIQUE KEY unique_advisor (group_id, user_id),
            INDEX idx_group_user (group_id, user_id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci");
        
        // جدول ویژه‌ها
        $db->exec("CREATE TABLE IF NOT EXISTS mir_vips (
            id INT AUTO_INCREMENT PRIMARY KEY,
            group_id BIGINT NOT NULL,
            user_id BIGINT NOT NULL,
            user_name VARCHAR(255),
            username VARCHAR(255),
            added_by BIGINT NOT NULL,
            added_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            UNIQUE KEY unique_vip (group_id, user_id),
            INDEX idx_group_user (group_id, user_id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci");
        
        // جدول بن سراسری
        $db->exec("CREATE TABLE IF NOT EXISTS mir_global_banned (
            id INT AUTO_INCREMENT PRIMARY KEY,
            user_id BIGINT NOT NULL UNIQUE,
            user_name VARCHAR(255),
            username VARCHAR(255),
            banned_by BIGINT NOT NULL,
            banned_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            INDEX idx_user_id (user_id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci");
        
        // جدول دسترسی‌ها (برای پنل)
        $db->exec("CREATE TABLE IF NOT EXISTS mir_permissions (
            id INT AUTO_INCREMENT PRIMARY KEY,
            group_id BIGINT NOT NULL,
            user_id BIGINT NOT NULL,
            can_ban TINYINT(1) DEFAULT 0,
            can_mute TINYINT(1) DEFAULT 0,
            can_delete TINYINT(1) DEFAULT 0,
            can_pin TINYINT(1) DEFAULT 0,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            UNIQUE KEY unique_permission (group_id, user_id),
            INDEX idx_group_user (group_id, user_id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci");
        
        // جدول داده‌های موقت
        $db->exec("CREATE TABLE IF NOT EXISTS mir_temp_data (
            id INT AUTO_INCREMENT PRIMARY KEY,
            group_id BIGINT NOT NULL,
            user_id BIGINT NOT NULL,
            data_key VARCHAR(100) NOT NULL,
            data_value TEXT,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            UNIQUE KEY unique_temp (group_id, user_id, data_key),
            INDEX idx_created_at (created_at)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci");
        
        // جدول اطلاعات کاربران برای MIR
        $db->exec("CREATE TABLE IF NOT EXISTS mir_users (
            id INT AUTO_INCREMENT PRIMARY KEY,
            group_id BIGINT NOT NULL,
            user_id BIGINT NOT NULL,
            first_name VARCHAR(255),
            username VARCHAR(255),
            last_seen DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            UNIQUE KEY unique_user (group_id, user_id),
            INDEX idx_group_user (group_id, user_id),
            INDEX idx_username (username)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci");
    }
    // اجرای ساخت جداول
    createMirTables();
}
?>