<?php
// ═══════════════════════════════════════════════════════
// 📅 فایل jdate - تبدیل تاریخ میلادی به شمسی
// ═══════════════════════════════════════════════════════

/**
 * تبدیل تاریخ میلادی به شمسی (جلالی)
 * 
 * @param string $format فرمت خروجی (مشابه date در PHP)
 * @param int|null $timestamp تایم‌استمپ یونیکس (null = زمان فعلی)
 * @return string تاریخ شمسی
 */
function jdate($format, $timestamp = null) {
    if ($timestamp === null) {
        $timestamp = time();
    }
    
    // تبدیل تاریخ میلادی به شمسی
    list($jy, $jm, $jd) = gregorian_to_jalali(
        date('Y', $timestamp),
        date('m', $timestamp),
        date('d', $timestamp)
    );
    
    // نام‌های فارسی
    $daysOfWeek = [
        'یکشنبه',
        'دوشنبه',
        'سه‌شنبه',
        'چهارشنبه',
        'پنج‌شنبه',
        'جمعه',
        'شنبه'
    ];
    
    $monthsName = [
        1 => 'فروردین',
        2 => 'اردیبهشت',
        3 => 'خرداد',
        4 => 'تیر',
        5 => 'مرداد',
        6 => 'شهریور',
        7 => 'مهر',
        8 => 'آبان',
        9 => 'آذر',
        10 => 'دی',
        11 => 'بهمن',
        12 => 'اسفند'
    ];
    
    // جایگزینی فرمت‌ها
    $output = $format;
    
    // l = نام روز هفته
    $output = str_replace('l', $daysOfWeek[date('w', $timestamp)], $output);
    
    // D = نام کوتاه روز هفته
    $dayShort = mb_substr($daysOfWeek[date('w', $timestamp)], 0, 1);
    $output = str_replace('D', $dayShort, $output);
    
    // j = روز ماه بدون صفر
    $output = str_replace('j', $jd, $output);
    
    // d = روز ماه با صفر
    $output = str_replace('d', sprintf('%02d', $jd), $output);
    
    // F = نام کامل ماه
    $output = str_replace('F', $monthsName[$jm], $output);
    
    // M = نام کوتاه ماه
    $monthShort = mb_substr($monthsName[$jm], 0, 3);
    $output = str_replace('M', $monthShort, $output);
    
    // m = شماره ماه با صفر
    $output = str_replace('m', sprintf('%02d', $jm), $output);
    
    // n = شماره ماه بدون صفر
    $output = str_replace('n', $jm, $output);
    
    // Y = سال چهار رقمی
    $output = str_replace('Y', $jy, $output);
    
    // y = سال دو رقمی
    $output = str_replace('y', substr($jy, -2), $output);
    
    // H = ساعت 24 با صفر
    $output = str_replace('H', date('H', $timestamp), $output);
    
    // h = ساعت 12 با صفر
    $output = str_replace('h', date('h', $timestamp), $output);
    
    // G = ساعت 24 بدون صفر
    $output = str_replace('G', date('G', $timestamp), $output);
    
    // g = ساعت 12 بدون صفر
    $output = str_replace('g', date('g', $timestamp), $output);
    
    // i = دقیقه با صفر
    $output = str_replace('i', date('i', $timestamp), $output);
    
    // s = ثانیه با صفر
    $output = str_replace('s', date('s', $timestamp), $output);
    
    // A = AM/PM بزرگ
    $output = str_replace('A', date('A', $timestamp), $output);
    
    // a = am/pm کوچک
    $output = str_replace('a', date('a', $timestamp), $output);
    
    return $output;
}

/**
 * تبدیل تاریخ میلادی به شمسی
 * 
 * @param int $gy سال میلادی
 * @param int $gm ماه میلادی
 * @param int $gd روز میلادی
 * @return array [سال، ماه، روز] شمسی
 */
function gregorian_to_jalali($gy, $gm, $gd) {
    $g_d_m = [0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334];
    
    if ($gy > 1600) {
        $jy = 979;
        $gy -= 1600;
    } else {
        $jy = 0;
        $gy -= 621;
    }
    
    $gy2 = ($gm > 2) ? ($gy + 1) : $gy;
    $days = (365 * $gy) + (int)(($gy2 + 3) / 4) - (int)(($gy2 + 99) / 100) + (int)(($gy2 + 399) / 400) - 80 + $gd + $g_d_m[$gm - 1];
    $jy += 33 * (int)($days / 12053);
    $days %= 12053;
    $jy += 4 * (int)($days / 1461);
    $days %= 1461;
    
    if ($days > 365) {
        $jy += (int)(($days - 1) / 365);
        $days = ($days - 1) % 365;
    }
    
    if ($days < 186) {
        $jm = 1 + (int)($days / 31);
        $jd = 1 + ($days % 31);
    } else {
        $jm = 7 + (int)(($days - 186) / 30);
        $jd = 1 + (($days - 186) % 30);
    }
    
    return [$jy, $jm, $jd];
}

/**
 * تبدیل تاریخ شمسی به میلادی
 * 
 * @param int $jy سال شمسی
 * @param int $jm ماه شمسی
 * @param int $jd روز شمسی
 * @return array [سال، ماه، روز] میلادی
 */
function jalali_to_gregorian($jy, $jm, $jd) {
    if ($jy > 979) {
        $gy = 1600;
        $jy -= 979;
    } else {
        $gy = 621;
    }
    
    $days = (365 * $jy) + ((int)($jy / 33) * 8) + (int)(($jy % 33 + 3) / 4) + 78 + $jd + (($jm < 7) ? ($jm - 1) * 31 : (($jm - 7) * 30) + 186);
    $gy += 400 * (int)($days / 146097);
    $days %= 146097;
    
    if ($days > 36524) {
        $gy += 100 * (int)(--$days / 36524);
        $days %= 36524;
        if ($days >= 365) $days++;
    }
    
    $gy += 4 * (int)($days / 1461);
    $days %= 1461;
    
    if ($days > 365) {
        $gy += (int)(($days - 1) / 365);
        $days = ($days - 1) % 365;
    }
    
    $sal_a = [0, 31, (($gy % 4 == 0 && $gy % 100 != 0) || ($gy % 400 == 0)) ? 29 : 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31];
    
    $gm = 0;
    while ($gm < 13 && $days > $sal_a[$gm]) {
        $days -= $sal_a[$gm++];
    }
    
    $gd = $days;
    
    return [$gy, $gm, $gd];
}

/**
 * دریافت زمان فعلی به فرمت شمسی
 * 
 * @param string $format فرمت خروجی
 * @return string تاریخ و زمان فعلی شمسی
 */
function jnow($format = 'Y/m/d H:i:s') {
    return jdate($format);
}
?>